#include "PreviewWindow.h"
#include "src/gui_elements/Atomic.h"
#include "src/main/colors.h"
#include "src/main/dimensions.h"
#include "src/main/inishell.h"

#include <QMenuBar>

SyntaxHighlighter::SyntaxHighlighter(QTextDocument *textdoc) : QSyntaxHighlighter(textdoc)
{
	HighlightingRule rule;

	QTextCharFormat format_section;
	format_section.setForeground(colors::getQColor("syntax_unknown_section"));
	format_section.setFontWeight(QFont::Bold);
	rule.pattern = QRegularExpression(R"(.*\)" + Cst::section_open + R"(.*\)" + Cst::section_close + R"(.*)");
	rule.format = format_section;
	rules_.append(rule);

	QTextCharFormat format_unknown_key;
	format_unknown_key.setForeground(colors::getQColor("syntax_unknown_key"));
	rule.pattern = QRegularExpression(R"(^\s*\w+(?=\s*=))");
	rule.format = format_unknown_key;
	rules_.append(rule);

	QTextCharFormat format_value;
	format_value.setForeground(Qt::blue);
	rule.pattern = QRegularExpression(R"((?<=\=).*)");
	rule.format = format_value;
	rules_.append(rule);

	/* populate highlighter with known sections and keys */
	QTextCharFormat format_known_key;
	format_known_key.setForeground(colors::getQColor("syntax_known_key"));
	QTextCharFormat format_known_section;
	format_known_section.setForeground(colors::getQColor("syntax_known_section"));
	format_known_section.setFontWeight(QFont::Bold);

	QList<Atomic *> panel_list = getMainWindow()->findChildren<Atomic *>();
	for (auto &panel : panel_list) {
		if (panel->property("no_ini").toBool() == true)
			continue;
		QString value, section, key;
		value = panel->getIniValue(section, key); //TODO: escape only if needed for the set char
		rule.pattern = QRegularExpression("\\" + Cst::section_open + section + "\\" +
		    Cst::section_close, QRegularExpression::CaseInsensitiveOption);
		rule.format = format_known_section;
		rules_.append(rule);
		rule.pattern = QRegularExpression(R"(^\s*)" + key + R"((=|\s))", QRegularExpression::CaseInsensitiveOption);
		rule.format = format_known_key;
		rules_.append(rule);
	}

	QTextCharFormat format_equal;
	format_equal.setForeground(Qt::black);
	rule.pattern = QRegularExpression(R"(=)");
	rule.format = format_equal;
	rules_.append(rule);
}

void SyntaxHighlighter::highlightBlock(const QString &text)
{
	for (const HighlightingRule &rule : qAsConst(rules_)) {
		QRegularExpressionMatchIterator mit = rule.pattern.globalMatch(text);
		while (mit.hasNext()) {
			QRegularExpressionMatch match = mit.next();
			setFormat(match.capturedStart(), match.capturedLength(), rule.format);
		}
	}
}

PreviewWindow::PreviewWindow(QMainWindow *parent) : QMainWindow(parent)
{
	file_tabs_ = new QTabWidget;
	connect(file_tabs_, &QTabWidget::tabCloseRequested, this, &PreviewWindow::closeTab);
	file_tabs_->setTabsClosable(true);
	this->setCentralWidget(file_tabs_);
	createMenu();

	dim::setDimensions(this, dim::PREVIEW);
}

void PreviewWindow::addIniTab()
{
	auto *preview_editor = new QTextEdit;
	preview_editor->setStyleSheet("QTextEdit {background-color: " + colors::getQColor("syntax_background").name() +
	    "; color: " + colors::getQColor("syntax_invalid").name() + "}");
	highlighter_ = new SyntaxHighlighter(preview_editor->document());

	QString ini_contents;
	QTextStream ss(&ini_contents);

	INIParser gui_ini = getMainWindow()->getIniCopy();
	getMainWindow()->getControlPanel()->setIniValuesFromGui(&gui_ini);

	gui_ini.outputIni(ss);
	preview_editor->setPlainText(ini_contents);
	QFileInfo file_info(getMainWindow()->getIni()->getFilename());
	file_tabs_->addTab(preview_editor, file_info.fileName() + " *");
	file_tabs_->setTabToolTip(file_tabs_->count() - 1, file_info.filePath());
	file_tabs_->setCurrentIndex(file_tabs_->count() - 1);
}

void PreviewWindow::closeTab(int index)
{
	file_tabs_->removeTab(index);
	if (file_tabs_->count() == 0)
		this->close();
}

void PreviewWindow::createMenu()
{
	/* File menu */
	QMenu *menu_file = this->menuBar()->addMenu(tr("&File"));
	auto *file_save = new QAction(tr("&Save"));
	menu_file->addAction(file_save);
	connect(file_save, &QAction::triggered, this, &PreviewWindow::saveFile);
	auto *file_save_as = new QAction(tr("Save &as..."));
	menu_file->addAction(file_save_as);
	connect(file_save_as, &QAction::triggered, this, &PreviewWindow::saveFileAs);
}

void PreviewWindow::saveFile()
{
	std::cout << "save" << std::endl;
}

void PreviewWindow::saveFileAs()
{
	std::cout << "save as" << std::endl;
}
