/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * The main program window.
 * Every widget that has this as ancestor will be destroyed when the application quits.
 * 2019-10
 */

#ifndef MAINWINDOW_H
#define MAINWINDOW_H

#include "Logger.h"
#include "src/gui/MainPanel.h"
#include "src/gui/PreviewWindow.h"
#include "src/main/constants.h"
#include "src/main/INIParser.h"

#include <QAction>
#include <QCheckBox>
#include <QIcon>
#include <QLabel>
#include <QList>
#include <QMainWindow>
#include <QString>
#include <QTabBar>
#include <QTimer>
#include <QtXml>
#include <QWidgetList>


class MouseEventFilter : public QObject {
	public:
		bool eventFilter(QObject *object, QEvent *event) override;
};

class MainWindow : public QMainWindow {
	Q_OBJECT //Qt macro to make code g++ ready

	public:
		explicit MainWindow(QDomDocument &xml_settings, QMainWindow *parent = nullptr);
		~MainWindow();
		void buildGui(const QDomDocument &xml);
		MainPanel * getControlPanel() const { return control_panel_; }
		QList<Atomic *> getPanelsForKey(const QString &ini_key);
		void setStatus(const QString &message, const QString &color = "black", const bool &status_light = false,
		    const int &time = -1);
		void setStatusLight(const bool &on);
		void refreshStatus();
		void log(const QString &message, const QString &color = "black") { logger_.log(message, color); }
		INIParser * getIni() { return &ini_; }
		INIParser getIniCopy() { return ini_; }
		void openIni(const QString &path, const bool &is_autoopen = false);
		void openXml(const QString &path, const QString &app_name, const bool &fresh = true);
		QDomDocument * getXmlSettings() const { return &xml_settings_; }

	public slots:
		void viewLogger();

	private:
		void createMenu();
		void createToolbar();
		void createStatusbar();
		QWidgetList findPanel(QWidget *parent, const Section &section, const KeyValue &keyval);
		QWidgetList findSimplePanel(QWidget *parent, const Section &section, const KeyValue &keyval);
		QWidgetList prepareSelector(QWidget *parent, const Section &section, const KeyValue &keyval);
		QWidgetList prepareReplicator(QWidget *parent, const Section &section, const KeyValue &keyval);
		bool setGuiFromIni(const INIParser &ini);
		void saveIni(const QString &filename = QString());
		void saveIniAs();
		void openIni();
		void closeIni();
		void clearGui();

		QToolBar *toolbar_ = nullptr; //init against warnings
		QAction *toolbar_open_ini_ = nullptr;
		QAction *toolbar_clear_gui_ = nullptr;
		QAction *toolbar_save_ini_ = nullptr;
		QAction *toolbar_save_ini_as_ = nullptr;
		QAction *toolbar_preview_ = nullptr;
		MainPanel *control_panel_ = nullptr;
		PreviewWindow *preview_ = nullptr;
		Logger logger_;
		INIParser ini_;
		QDomDocument &xml_settings_;
		QLabel *status_label_ = nullptr;
		QLabel *status_icon_ = nullptr;
		QTimer status_timer_;
		QLabel *ini_filename_ = nullptr;
		QCheckBox *autoload_box_ = nullptr;
		QAction *autoload_ = nullptr;

	private slots:
		void clearStatus();
		void quitProgram();
		void viewPreview();
		void viewSettings();
		void loadHelp();
		void helpAbout();
		void toolbarClick(const QString &function);
		void onAutoloadCheck(const int &state);

	friend class MouseEventFilter;
};

#endif //MAINWINDOW_H
