/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * The main window that is displayed on startup.
 * 2019-10
 */

#include "MainWindow.h"
#include "src/gui/AboutWindow.h"
#include "src/gui/Settings.h"
#include "src/gui_elements/Group.h" //to exclude Groups from panel search
#include "src/main/colors.h"
#include "src/main/Error.h"
#include "src/main/dimensions.h"
#include "src/main/INIParser.h"
#include "src/main/inishell.h"
#include "src/main/XMLReader.h"

#include <QApplication>
#include <QCheckBox>
#include <QDesktopServices>
#include <QFileDialog>
#include <QGroupBox>
#include <QMenuBar>
#include <QMessageBox>
#include <QRegularExpression>
#include <QSpacerItem>
#include <QStatusBar>
#include <QSysInfo>
#include <QToolBar>

#ifdef DEBUG
	#include <iostream>
#endif

bool MouseEventFilter::eventFilter(QObject *object, QEvent *event)
{
	if (event->type() == QEvent::MouseButtonPress) {
		if (object->property("mouseclick") == "open_log")
			getMainWindow()->viewLogger();
		else if (object->property("mouseclick") == "open_ini")
			QDesktopServices::openUrl("file:" + getMainWindow()->ini_filename_->text());
	}
	return QObject::eventFilter(object, event);
}

/**
 * @class MainWindow
 * @brief Constructor for the main window.
 * @param[in] xml_settings Settings for the static INIshell GUI.
 * @param[in] parent Unused since the main window is kept in scope by the entry point function.
 */
MainWindow::MainWindow(QDomDocument &xml_settings, QMainWindow *parent)
    : QMainWindow(parent), logger_(this), xml_settings_(xml_settings)
{
	logger_.logSystemInfo();

	/* retrieve and set main window geometry */
	dim::setDimensions(this, dim::MAIN_WINDOW);
	dim::setDimensions(&logger_, dim::LOGGER);

	/* create basic GUI items */
	createMenu();
	createToolbar();
	createStatusbar();
	preview_ = new PreviewWindow(this);
	/* create the dynamic GUI area */
	control_panel_ = new MainPanel;
	this->setCentralWidget(control_panel_);
	ini_.setLogger(&logger_);

	setStatus("Ready.", "sysinfo");
//	const QString xml_file("applications/meteoio_config.xml");
////	const QString xml_file("min.xml");

//	} else {
//		Error(tr("Last opened XML file not found."), tr("Input file \"%1\" does not exist").arg(xml_file));
//	}
}

MainWindow::~MainWindow()
{
	saveSettings(xml_settings_);
}

/**
 * @brief Build the dynamic GUI.
 * This function initiates the recursive GUI building with an XML document that was
 * parsed beforehand.
 * @param[in] xml XML to build the gui from.
 */
void MainWindow::buildGui(const QDomDocument &xml)
{
	setUpdatesEnabled(false); //disable painting until done
	QDomNode root = xml.firstChild();
	while (!root.isNull()) {
		if (root.isElement()) { //skip over comments
			//give no parent group - tabs will be created for top level:
			recursiveBuild(root, nullptr, QString());
			break;
		}
		root = root.nextSibling();
	}
	setUpdatesEnabled(true);
}

bool MainWindow::setGuiFromIni(const INIParser &ini)
{
	for (auto &sec : ini.getSections())
	{
		ScrollPanel *tab_scroll = getControlPanel()->getSectionScrollarea(sec.getName(), QString(), QString(), true);
		if (tab_scroll != nullptr) { //section exists on GUI
			for (auto &kv : sec.getKeyValueList() ) {
				QWidgetList widgets = findPanel(tab_scroll, sec, kv.second);
				if (!widgets.isEmpty()) {
					for (int ii = 0; ii < widgets.size(); ++ii) //multiple panels can share the same key
						widgets.at(ii)->setProperty("ini_value", kv.second.getValue());
				} else {
					logger_.log(tr("No GUI element found for INI key \"") + sec.getName() + Cst::sep + kv.second.getKey() + "\"",
					    "iniwarning");
					topStatus(tr("Encountered unknown INI key"), "iniwarning");
				}
			} //endfor kv
		} else { //section does not exist
			log(tr("No matching section in GUI for section from INI file: \"") +
			    sec.getName() + "\"", "iniwarning");
			topStatus(tr("Encountered unknown INI key"), "iniwarning");
		} //endif section exists
	} //endfor it
	return true;
}

QList<Atomic *> MainWindow::getPanelsForKey(const QString &ini_key)
{
	QList<Atomic *> panel_list = control_panel_->findChildren<Atomic *>(Atomic::getQtKey(ini_key));
	for (auto it = panel_list.begin(); it != panel_list.end(); ++it) {
		if (qobject_cast<Group *>(*it))
			panel_list.erase(it); //Groups don't count towards finding INI keys
	}
	return panel_list;
}

void MainWindow::saveIni(const QString &filename)
{
	INIParser gui_ini = ini_;
	const QString missing = control_panel_->setIniValuesFromGui(&gui_ini);
	if (!missing.isEmpty()) {
		QMessageBox msgMissing;
		msgMissing.setWindowTitle("Warning ~ " + QCoreApplication::applicationName());
		msgMissing.setText(tr("<b>Missing mandatory INI values.</b>"));
		msgMissing.setInformativeText(tr("Some non-optional INI keys are not set.\nSee details for a list or go back to the GUI and set all highlighted fields."));
		msgMissing.setDetailedText(tr("Missing INI keys: \n") + missing);
		msgMissing.setIcon(QMessageBox::Warning);
		msgMissing.setStandardButtons(QMessageBox::Save | QMessageBox::Cancel);
		msgMissing.setDefaultButton(QMessageBox::Cancel);
		int clicked = msgMissing.exec();
		if (clicked == QMessageBox::Cancel)
			return;
	}
	gui_ini.writeIni(filename.isEmpty()? gui_ini.getFilename() : filename);
}

void MainWindow::saveIniAs()
{
	const QString filename = QFileDialog::getSaveFileName(this, tr("Save INI file"), "./",
	    "INI files (*.ini);;All files (*)", nullptr, QFileDialog::DontUseNativeDialog);
	if (filename.isNull()) //cancelled
		return;
	saveIni(filename);
	ini_.setFilename(filename);
	ini_filename_->setText(filename);
	autoload_->setVisible(true);
	toolbar_save_ini_->setEnabled(true);
}

void MainWindow::openIni()
{
	const QString path = QFileDialog::getOpenFileName(this, tr("Open INI file"), "./",
	    "INI files (*.ini);;All files (*)", nullptr, QFileDialog::DontUseNativeDialog | QFileDialog::DontConfirmOverwrite);
	if (path.isNull()) //cancelled
		return;
	openIni(path);
}

void MainWindow::openIni(const QString &path, const bool &is_autoopen)
{
	ini_.setFile(path);
	setGuiFromIni(ini_);
	toolbar_save_ini_->setEnabled(true);
	toolbar_preview_->setEnabled(true);
	autoload_->setVisible(true);
	ini_filename_->setText(path);
	autoload_box_->setProperty("xml_application", control_panel_->getWorkflowPanel()->getCurrentApplication());
	autoload_box_->setText(tr("autoload for ") + autoload_box_->property("xml_application").toString());
	if (!is_autoopen)
		autoload_box_->setCheckState(Qt::Unchecked);
}

void MainWindow::closeIni()
{
	toolbar_save_ini_->setEnabled(false);
	toolbar_preview_->setEnabled(false);
	ini_filename_->setText(QString());
}

void MainWindow::clearGui()
{
	ini_.clear();
	getControlPanel()->clearGui();
	closeIni();
	ini_filename_->setText(QString());
	autoload_->setVisible(false);
//	toolbar_clear_gui_->setEnabled(false);
}

void MainWindow::openXml(const QString &path, const QString &app_name, const bool &fresh)
{
	if (fresh)
		control_panel_->clearGuiElements();

	if (QFile::exists(path)) {
		XMLReader xml;
		QString xml_error = QString();
		xml.read(path, xml_error);
		if (!xml_error.isNull()) {
			xml_error.chop(1); //trailing \n
			Error(tr("Errors occured when parsing the XML configuration file"),
			    tr("File: \"") + path + "\"", xml_error);
		}
		setStatus("Building GUI...", "black", true);
		buildGui(xml.getXml());
		setStatus("Ready.", "sysinfo", false);
		control_panel_->getWorkflowPanel()->buildWorkflowPanel(xml.getXml());
	} else {
		topStatus(tr("File has been removed"));
	}

	toolbar_save_ini_as_->setEnabled(true);
	autoload_box_->setText(tr("autoload for ") + app_name);
	autoload_box_->setProperty("xml_application", app_name);

	bool found_autoload = false;
	for (auto ini_node = xml_settings_.firstChildElement().firstChildElement("auto").firstChildElement("autoload").firstChildElement("ini");
	    !ini_node.isNull(); ini_node = ini_node.nextSiblingElement("ini")) {
		if (ini_node.attribute("application").toLower() == app_name.toLower()) {
			autoload_box_->blockSignals(true); //don't re-save the setting we just fetched
			autoload_box_->setCheckState(Qt::Checked);
			autoload_box_->blockSignals(false);
			openIni(ini_node.text(), true);
			found_autoload = true;
			break;
		}
	}
	if (!found_autoload)
		autoload_box_->setCheckState(Qt::Unchecked);
	toolbar_clear_gui_->setEnabled(true);
	toolbar_open_ini_->setEnabled(true);
	getControlPanel()->getWorkflowPanel()->getFilesystemView()->setEnabled(true);
}

QWidgetList MainWindow::findPanel(QWidget *parent, const Section &section, const KeyValue &keyval)
{
	QWidgetList panels;
	panels = findSimplePanel(parent, section, keyval);
	int panel_count = parent->findChildren<Atomic *>().size();
	if (panels.isEmpty()) {
		panels = prepareSelector(parent, section, keyval); //TODO: logic cleanup
	}
	if (panels.isEmpty()) {
		panels = prepareReplicator(parent, section, keyval);
	}
	if (parent->findChildren<Atomic *>().size() != panel_count) //new elements were created
		return findPanel(parent, section, keyval);
	else
		return panels;
}

QWidgetList MainWindow::findSimplePanel(QWidget *parent, const Section &section, const KeyValue &keyval)
{
	QString id(section.getName() + Cst::sep + keyval.getKey());
	/* simple, not nested, keys */
	return parent->findChildren<QWidget *>(Atomic::getQtKey(id));
}

QWidgetList MainWindow::prepareSelector(QWidget *parent, const Section &section, const KeyValue &keyval)
{
	QString id(section.getName() + Cst::sep + keyval.getKey());
	const QString regex_selector("^" + section.getName() + Cst::sep + R"((\w+)(::)(\w+?)([0-9]*$))");
	const QRegularExpression rex(regex_selector);
	const QRegularExpressionMatch matches = rex.match(id);

	if (matches.captured(0) == id) { //it's from a selector with template children
		static const size_t idx_parameter = 1;
		static const size_t idx_keyname = 3;
		static const size_t idx_optional_number = 4;
		const QString parameter(matches.captured(idx_parameter));
		const QString key_name(matches.captured(idx_keyname));
		const QString number(matches.captured(idx_optional_number));
		QString gui_id = section.getName() + Cst::sep + "%" + Cst::sep + key_name + (number.isEmpty()? "" : "#");

		//try to find selector:
		QList<Selector *> selector_list = parent->findChildren<Selector *>(Atomic::getQtKey(gui_id));
		for (int ii = 0; ii < selector_list.size(); ++ii)
			selector_list.at(ii)->setProperty("ini_value", parameter); //cf. notes in prepareReplicator
		//now all the right child widgets should exist and we can try to find again:
		const QString key_id = section.getName() + Cst::sep + keyval.getKey(); //the one we were originally looking for
		return parent->findChildren<QWidget *>(Atomic::getQtKey(key_id));
	}
	return QWidgetList();
}

QWidgetList MainWindow::prepareReplicator(QWidget *parent, const Section &section, const KeyValue &keyval)
{
	QString id(section.getName() + Cst::sep + keyval.getKey());
	const QString regex_selector("^" + section.getName() + Cst::sep +
	    R"((\w+)" + Cst::sep + R"()*(\w*?)(?=\d)(\d*)$)");
	const QRegularExpression rex(regex_selector);
	const QRegularExpressionMatch matches = rex.match(id);

	if (matches.captured(0) == id) { //it's from a replicator with template children
		/*
		 * If we arrive here, we try to find the panel for an INI key that could belong
		 * to the child of a Replicator, e. g. "INPUT::STATION1" or "FILTERS::TA::FILTER1".
		 * We extract the name the Replicator would have (number to "#") and try finding it.
		 */
		static const size_t idx_parameter = 1;
		static const size_t idx_key = 2;
		static const size_t idx_number = 3;
		QString parameter(matches.captured(idx_parameter)); //has trailing "::" if existent
		const QString key(matches.captured(idx_key));
		const QString number(matches.captured(idx_number));
		QString gui_id = section.getName() + Cst::sep + parameter + key + "#"; //Replicator's name

		/*
		 * A replicator can't normally be accessed via INI keys, because there is no
		 * standalone XML code for it. It always occurs together with child panels,
		 * and those are the ones that will be sought by the INI parser to set values.
		 * Therefore we can use the "ini_value" property listener for a Replicator to
		 * tell it to replicate its panel, thus creating the necessary child panels.
		 */
		QList<Replicator *> replicator_list = parent-> //try to find replicator:
		    findChildren<Replicator *>(Atomic::getQtKey(gui_id));
		for (int ii = 0; ii < replicator_list.count(); ++ii)
			replicator_list.at(ii)->setProperty("ini_value", number);
		//now all the right child panels should exist and we can try to find again:
		const QString key_id(section.getName() + Cst::sep + keyval.getKey()); //the one we were originally looking for
		return parent->findChildren<QWidget *>(Atomic::getQtKey(key_id));
	} //endif has match

	return QWidgetList(); //no suitable Replicator found
}

/**
 * @brief Build the main window's menu items.
 */
void MainWindow::createMenu()
{
	/* File menu */
	QMenu *menu_file = this->menuBar()->addMenu(tr("&File"));
	auto *file_quit = new QAction(tr("&Exit"));
	menu_file->addAction(file_quit);
	connect(file_quit, &QAction::triggered, this, &MainWindow::quitProgram);

	/* View menu */
	QMenu *menu_view = this->menuBar()->addMenu(tr("&View"));
	auto *view_preview = new QAction(tr("&Preview"));
	menu_view->addAction(view_preview);
	connect(view_preview, &QAction::triggered, this, &MainWindow::viewPreview);
	auto *view_log = new QAction(tr("&Log"));
	menu_view->addAction(view_log);
	connect(view_log, &QAction::triggered, this, &MainWindow::viewLogger);
	auto *view_settings = new QAction(tr("&Settings"));
	menu_view->addAction(view_settings);
	connect(view_settings, &QAction::triggered, this, &MainWindow::viewSettings);

	/* Help menu */
	QMenuBar *menu_help_main = new QMenuBar(this->menuBar());
	QMenu *menu_help = menu_help_main->addMenu(tr("&Help"));
	auto *help = new QAction(tr("&Help"));
	menu_help->addAction(help);
	connect(help, &QAction::triggered, this, &MainWindow::loadHelp);
	auto *help_about = new QAction(tr("&About"));
	menu_help->addAction(help_about);
	connect(help_about, &QAction::triggered, this, &MainWindow::helpAbout);
	menu_help->addSeparator();
	auto *help_bugreport = new QAction(tr("File &bug report..."));
	menu_help->addAction(help_bugreport);
	connect(help_bugreport, &QAction::triggered,
	    [=]{ QDesktopServices::openUrl(QUrl("https://models.slf.ch/p/inishell-ng/issues/")); });

	this->menuBar()->setCornerWidget(menu_help_main); //push help menu to the right
}

/**
 * @brief Create the main window's toolbar.
 */
void MainWindow::createToolbar()
{
	//TODO: store position
	toolbar_ = this->addToolBar("main");
	toolbar_->setIconSize(QSize(32, 32));
	QPixmap icon_file;
	icon_file.load(":/icons/ini_file.png");
	toolbar_open_ini_ = toolbar_->addAction(QIcon(icon_file), tr("Open INI file"));
	connect(toolbar_open_ini_, &QAction::triggered, this, [=] { toolbarClick("open_ini"); });
	toolbar_open_ini_->setEnabled(false);
	toolbar_->addSeparator();
	icon_file.load(":/icons/save.png");
	toolbar_save_ini_ = toolbar_->addAction(QIcon(icon_file), tr("Save INI"));
	toolbar_save_ini_->setEnabled(false); //enable when an INI is open
	connect(toolbar_save_ini_, &QAction::triggered, this, [=] { toolbarClick("save_ini"); });
	icon_file.load(":/icons/save_as.png");
	toolbar_save_ini_as_ = toolbar_->addAction(QIcon(icon_file), tr("Save INI file as"));
	toolbar_save_ini_as_->setEnabled(false);
	connect(toolbar_save_ini_as_, &QAction::triggered, this, [=] { toolbarClick("save_ini_as"); });
	icon_file.load(":/icons/preview.png");
	toolbar_preview_ = toolbar_->addAction(QIcon(icon_file), tr("Preview INI"));
	toolbar_preview_->setEnabled(false);
	connect(toolbar_preview_, &QAction::triggered, this, [=] { toolbarClick("preview"); });
	toolbar_->addSeparator();
	icon_file.load(":/icons/clear.png");
	toolbar_clear_gui_ = toolbar_->addAction(QIcon(icon_file), tr("Clear INI settings"));
	connect(toolbar_clear_gui_, &QAction::triggered, this, [=] { toolbarClick("clear_gui"); });
	toolbar_clear_gui_->setEnabled(false);

	QWidget *spacer = new QWidget;
	spacer->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);
	QWidget *small_spacer = new QWidget;
	small_spacer->setFixedWidth(25);
	ini_filename_ = new QLabel();
	ini_filename_->setStyleSheet("QLabel {color: " + colors::getQColor("sysinfo").name() + "}");
	ini_filename_->setProperty("mouseclick", "open_ini");
	auto *mouse_events = new MouseEventFilter;
	ini_filename_->installEventFilter(mouse_events);
	ini_filename_->setCursor(QCursor(Qt::PointingHandCursor));
	autoload_box_= new QCheckBox();

	toolbar_->addWidget(spacer);
	toolbar_->addWidget(ini_filename_);
	toolbar_->addWidget(small_spacer);
	autoload_box_ = new QCheckBox();
	connect(autoload_box_, &QCheckBox::stateChanged, this, &MainWindow::onAutoloadCheck);

	autoload_ = toolbar_->addWidget(autoload_box_);
	autoload_->setVisible(false);
	toolbar_->addAction(autoload_);
}

void MainWindow::createStatusbar()
{
	auto *spacer_widget = new QWidget;
	spacer_widget->setFixedSize(5, 0);
	status_label_ = new QLabel;
	status_label_->setProperty("mouseclick", "open_log");
	auto *mouse_events = new MouseEventFilter;
	status_label_->installEventFilter(mouse_events);
	status_label_->setCursor(QCursor(Qt::PointingHandCursor));
	status_icon_ = new QLabel;
	statusBar()->addWidget(spacer_widget);
	statusBar()->addWidget(status_label_);
	statusBar()->addPermanentWidget(status_icon_);
}

/**
 * @brief Display a text in the main window's status bar.
 * @details Events such as hovering over a menu may clear the status. It is meant
 * for temporary messages.
 * @param[in] message The text to display.
 * @param[in] time Time to display the text for.
 */
void MainWindow::setStatus(const QString &message, const QString &color, const bool &status_light, const int &time)
{ //only temporary messages are possible via statusBar()->showMessage
	status_label_->setText(message);
	status_label_->setStyleSheet("QLabel {color: " + colors::getQColor(color).name() + "}");
	setStatusLight(status_light);
	if (time > 0) {
		status_timer_.stop();
		status_timer_.singleShot(time, this, &MainWindow::clearStatus);
	}
}

void MainWindow::setStatusLight(const bool &on)
{
	const QPixmap icon((on? ":/icons/active.png" : ":/icons/inactive.png"));
	status_icon_->setPixmap(icon.scaled(15, 15));
}

void MainWindow::refreshStatus()
{
	status_label_->adjustSize();
	status_label_->repaint();
}

void MainWindow::clearStatus()
{
	status_label_->setText(QString());
}

/**
 * @brief Event handler for the "File::Quit" menu: quit the main program.
 */
void MainWindow::quitProgram()
{
	QApplication::quit();
}

void MainWindow::viewPreview()
{
	preview_->addIniTab();
	preview_->show();
	preview_->raise();
}

/**
 * @brief Event handler for the "View::Log" menu: open the Logger window.
 */
void MainWindow::viewLogger()
{
	logger_.show(); //the logger is always kept in scope
	logger_.raise(); //bring to front
}

/**
 * @brief Event handler for the "View::Settings" menu: open the settings dialog.
 */
void MainWindow::viewSettings()
{
	static auto *settings_dialog = new Settings(this); //allow only once, load when asked
	settings_dialog->show();
	settings_dialog->raise();
}

void MainWindow::loadHelp()
{
	clearGui();
	openXml(":doc/help.xml", "Help");
}

void MainWindow::helpAbout()
{
	static auto *about = new AboutWindow(this);
	about->show();
	about->raise();
}

void MainWindow::toolbarClick(const QString &function)
{
	if (function == "save_ini")
		saveIni();
	else if (function =="save_ini_as")
		saveIniAs();
	else if (function == "open_ini")
		openIni();
	else if (function == "clear_gui")
		clearGui();
	else if (function == "preview")
		viewPreview();
}

void MainWindow::onAutoloadCheck(const int &state)
{
	const QString app = autoload_box_->property("xml_application").toString();
	QDomNode autoload_node = xml_settings_.firstChildElement().firstChildElement("auto").firstChildElement("autoload");
	for (auto ini = autoload_node.firstChildElement("ini"); !ini.isNull(); ini = ini.nextSiblingElement("ini")) {
		if (ini.attribute("application").toLower() == app.toLower()) {
			if (state == Qt::Checked) {
				ini.firstChild().setNodeValue(ini_filename_->text());
				return;
			} else {
				ini.parentNode().removeChild(ini);
				return;
			}
		}
	}

	//not found - create new settings node:
	if (state == Qt::Checked) {
		QDomNode ini = autoload_node.appendChild(xml_settings_.createElement("ini"));
		ini.toElement().setAttribute("application", app);
		ini.appendChild(xml_settings_.createTextNode(ini_filename_->text()));
	}
}
