#include "FolderView.h"
#include "src/main/constants.h"
#include "src/main/inishell.h"

#include <QHBoxLayout>
#include <QHeaderView>
#include <QToolButton>
#include <QVBoxLayout>

#ifdef DEBUG
	#include <iostream>
#endif

FolderView::FolderView(QWidget *parent) : QWidget(parent)
{
	path_label_ = new QLabel;
	path_label_->setStyleSheet("QLabel {background-color: white}");
	filesystem_model_ = new QFileSystemModel;
	QStringList filters;
		filters << "*.ini";
	filesystem_model_->setNameFilters(filters);
	filesystem_model_->setRootPath("");
	path_label_->setText(".");
	filesystem_model_->setNameFilterDisables(false); //hide unfit items

	filesystem_tree_ = new QTreeView;
	connect(filesystem_tree_, &QTreeView::doubleClicked, this, &FolderView::onFilesysDoubleclicked);
	filesystem_tree_->setHorizontalScrollBarPolicy(Qt::ScrollBarAsNeeded);
	filesystem_tree_->header()->setSectionResizeMode(QHeaderView::Stretch); //expand columns
	filesystem_tree_->setIndentation(Cst::treeview_indentation_);
	filesystem_tree_->setHeaderHidden(true);
	filesystem_tree_->setModel(filesystem_model_);
	filesystem_tree_->setEnabled(false); //will get enabled if an XML is loaded
	for (int ii = 1; ii < filesystem_model_->columnCount(); ++ii)
		filesystem_tree_->hideColumn(ii); //show only name column

	const QString last_path = getMainWindow()->getXmlSettings()->firstChildElement().firstChildElement("auto").firstChildElement(
	    "history").firstChildElement("ini_folder").attribute("path");
	if (!last_path.isEmpty())
		setTreeIndex(filesystem_model_->index(last_path), true);

	filesystem_tree_->scrollTo(filesystem_model_->index(QDir::currentPath()));
	filesystem_tree_->expand(filesystem_model_->index(QDir::currentPath()));
	filesystem_tree_->setCurrentIndex(filesystem_model_->index(QDir::currentPath()));

	auto *button_back = new QToolButton;
	button_back->setIconSize(QSize(Cst::width_button_std, Cst::height_button_std));
	button_back->setAutoRaise(true);
	button_back->setIcon(QIcon(":/icons/folder_back.png"));
	connect(button_back, &QToolButton::clicked, this, &FolderView::onBackClicked);
	auto *button_up = new QToolButton;
	button_up->setIconSize(button_back->iconSize());
	button_up->setAutoRaise(true);
	button_up->setIcon(QIcon(":/icons/folder_up.png"));
	connect(button_up, &QToolButton::clicked, this, &FolderView::onUpClicked);
	auto *button_home = new QToolButton;
	button_home->setIconSize(button_back->iconSize());
	button_home->setAutoRaise(true);
	button_home->setIcon(QIcon(":/icons/folder_home.png"));
	connect(button_home, &QToolButton::clicked, this, &FolderView::onHomeClicked);
	auto *button_favorite = new QToolButton;
	button_favorite->setIconSize(button_back->iconSize());
	button_favorite->setAutoRaise(true);
	button_favorite->setIcon(QIcon(":/icons/folder_documents.png"));
	connect(button_favorite, &QToolButton::clicked, this, &FolderView::onFavoriteClicked);

	auto *toolbar_layout = new QHBoxLayout;
	toolbar_layout->setSpacing(0);
	toolbar_layout->addWidget(button_back);
	toolbar_layout->addWidget(button_up);
	toolbar_layout->addWidget(button_home);
	toolbar_layout->addWidget(button_favorite);
	toolbar_layout->addSpacerItem(new QSpacerItem(-1, -1, QSizePolicy::Expanding, QSizePolicy::Minimum));

	auto *main_layout = new QVBoxLayout;
	main_layout->addLayout(toolbar_layout);
	main_layout->addWidget(filesystem_tree_);
	main_layout->addWidget(path_label_);

	this->setLayout(main_layout);
}

void FolderView::setEnabled(const bool &enabled)
{
	filesystem_tree_->setEnabled(enabled);
}

void FolderView::setTreeIndex(const QModelIndex &index, const bool &no_stack)
{
	if (!no_stack) //back button click
		prev_index_stack_.push(filesystem_tree_->rootIndex());
	filesystem_tree_->setRootIndex(index);
	path_label_->setText(filesystem_model_->filePath(index));
}

void FolderView::onFilesysDoubleclicked(const QModelIndex &index)
{
	this->setProperty("previous_index", index.parent());
	auto info = QFileInfo(filesystem_model_->filePath(index));
	if (info.isDir()) {
		setTreeIndex(index);
		getMainWindow()->getXmlSettings()->firstChildElement().firstChildElement("auto").firstChildElement("history").firstChildElement(
		    "ini_folder").toElement().setAttribute("path", filesystem_model_->filePath(index));
	} else {
		getMainWindow()->openIni(info.filePath());
	}
}

void FolderView::onUpClicked()
{
	setTreeIndex(filesystem_tree_->rootIndex().parent());
}

void FolderView::onHomeClicked()
{
	setTreeIndex(filesystem_model_->index(""));
}

void FolderView::onBackClicked()
{
	if (!prev_index_stack_.isEmpty())
		setTreeIndex(prev_index_stack_.pop(), true);
}

void FolderView::onFavoriteClicked()
{
	setTreeIndex(filesystem_model_->index(QDir::currentPath()));
}
