#include "ApplicationsView.h"
#include "src/main/inishell.h"

#include <QAction>
#include <QDesktopServices>
#include <QFileDialog>
#include <QFileInfo>
#include <QRegularExpression>
#include <QVBoxLayout>

ApplicationsView::ApplicationsView(QWidget *parent) : QWidget(parent)
{
	application_list_ = new QListWidget;

	application_list_->setContextMenuPolicy(Qt::CustomContextMenu);
	createContextMenu();
	connect(application_list_, &QListWidget::customContextMenuRequested, this, &ApplicationsView::showListContextMenu);

	connect(application_list_, &QListWidget::itemDoubleClicked, this, &ApplicationsView::onListDoubleClick);
	auto *layout = new QVBoxLayout;
	layout->addWidget(application_list_);
	this->setLayout(layout);

	application_dir_.setPath("./applications");
	readAppsFromDir();
}

QString ApplicationsView::getCurrentApplication() const
{
	if (application_list_->currentRow() == -1)
		return QString();
	QListWidgetItem *current = application_list_->currentItem();
	return current->text();
}

void ApplicationsView::readAppsFromDir()
{
	application_list_->clear();
	QStringList apps = application_dir_.entryList(QStringList() << "*.xml" << "*.XML", QDir::Files);
	for (auto &filename : apps) {
		QFile infile(application_dir_.path() + "/" + filename);
		if (!infile.open(QIODevice::ReadOnly | QIODevice::Text)) {
			topLog(tr(R"(Could not check application file: unable to read "%1")").arg(filename),
			"fileerror");
			continue;
		}
		QTextStream tstream(&infile);
		while (!tstream.atEnd()) {
			QString line = tstream.readLine();
			QRegularExpression regex_inishell(R"(^\<inishell_config application=\"(.*?)\".*?(icon=\"(.*)\")*>$)");
			QRegularExpressionMatch match_inishell(regex_inishell.match(line));
			if (match_inishell.captured(0) == line && !line.isEmpty()) {
				addApplication(infile, match_inishell);
				break;
			}
		} //endwhile
	} //endfor filename
}

void ApplicationsView::addApplication(const QFile &file, const QRegularExpressionMatch &match_inishell)
{
	static const int idx_name = 1;
	static const int idx_icon = 3;
	auto *app = new QListWidgetItem;
	app->setText(match_inishell.captured(idx_name));
	app->setIcon(QIcon(application_dir_.path() + "/" + match_inishell.captured(idx_icon)));
	app->setData(Qt::UserRole, file.fileName());
	app->setToolTip(file.fileName());
	application_list_->addItem(app);
}

void ApplicationsView::createContextMenu()
{
	list_context_menu_.addAction(tr("Append to current GUI"));
	list_context_menu_.addSeparator();
	list_context_menu_.addAction(tr("Open in editor"));
	list_context_menu_.addAction(tr("Choose applications directory..."));
}

void ApplicationsView::onListDoubleClick(QListWidgetItem *item)
{
	getMainWindow()->openXml(item->data(Qt::UserRole).toString(), item->text());
	getMainWindow()->setWindowTitle(QCoreApplication::applicationName() + tr(" for ") + item->text());
}

void ApplicationsView::showListContextMenu(const QPoint &/*coords*/)
{
	QAction *selected = list_context_menu_.exec(QCursor::pos());
	if (selected) {
		if (selected->text().startsWith("Append") && application_list_->currentRow() != -1) {
			getMainWindow()->openXml(application_list_->currentItem()->data(Qt::UserRole).toString(),
			    application_list_->currentItem()->text(), false);
		} else if (selected->text().startsWith("Choose")) {
			const QString path = QFileDialog::getExistingDirectory(this, tr("Open Folder"), "./",
			    QFileDialog::DontUseNativeDialog | QFileDialog::ShowDirsOnly);
			if (!path.isNull()) {
				application_dir_.setPath(path);
				readAppsFromDir();
			}
		} else if (selected->text().startsWith("Open")) {
			QDesktopServices::openUrl(QUrl(application_list_->currentItem()->data(Qt::UserRole).toString()));
		}
	}
}
