/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with INIshell.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "PreviewWindow.h"
#include "src/gui_elements/Atomic.h"
#include "src/main/colors.h"
#include "src/main/common.h"
#include "src/main/dimensions.h"
#include "src/main/inishell.h"
#include "src/main/settings.h"

#include <vector>

#include <QCoreApplication>
#include <QFile>
#include <QFileDialog>
#include <QFileInfo>
#include <QFont>
#include <QFontDatabase>
#include <QKeySequence>
#include <QMenuBar>
#include <QMessageBox>
#include <QStatusBar>
#include <QTimer>

#ifdef DEBUG
	#include <iostream>
#endif //def DEBUG

/**
 * @class SyntaxHighlighter
 * @brief Default constructor for a syntax highlighter used in the INI preview.
 * @param[in] textdoc Text document to handle syntax highlighting for.
 */
SyntaxHighlighter::SyntaxHighlighter(QTextDocument *textdoc) : QSyntaxHighlighter(textdoc)
{
	HighlightingRule rule;

	/*
	 * The SyntaxHighlighter parses a document for INI file syntax and checks the sections
	 * and keys against the curently loaded XML. Sections and keys that are not present
	 * in the XML are highlighted differently.
	 */

	/* unknown sections */
	QTextCharFormat format_section;
	format_section.setForeground(colors::getQColor("syntax_unknown_section"));
	format_section.setFontWeight(QFont::Bold);
	rule.pattern = QRegularExpression(R"(.*\)" + Cst::section_open + R"(.*\)" + Cst::section_close + R"(.*)");
	rule.format = format_section;
	rules_.append(rule);

	/* unknown keys */
	QTextCharFormat format_unknown_key;
	format_unknown_key.setForeground(colors::getQColor("syntax_unknown_key"));
	rule.pattern = QRegularExpression(R"(^\s*\w+(?=\s*=))");
	rule.format = format_unknown_key;
	rules_.append(rule);

	/* INI values */
	QTextCharFormat format_value;
	format_value.setForeground(colors::getQColor("syntax_value"));
	rule.pattern = QRegularExpression(R"((?<=\=).*)");
	rule.format = format_value;
	rules_.append(rule);

	/* populate highlighter with known sections and keys */
	QTextCharFormat format_known_key;
	format_known_key.setForeground(colors::getQColor("syntax_known_key"));
	QTextCharFormat format_known_section;
	format_known_section.setForeground(colors::getQColor("syntax_known_section"));
	format_known_section.setFontWeight(QFont::Bold);

	const QList<Atomic *> panel_list( getMainWindow()->findChildren<Atomic *>() );
	for (auto &panel : panel_list) {
		if (panel->property("no_ini").toBool()) //e. g. Groups / frames
			continue;
		QString section, key;
		(void) panel->getIniValue(section, key);
		rule.pattern = QRegularExpression("\\" + Cst::section_open + section + "\\" +
		    Cst::section_close, QRegularExpression::CaseInsensitiveOption); //TODO: escape only if needed for the set char
		rule.format = format_known_section;
		rules_.append(rule);
		rule.pattern = QRegularExpression(R"(^\s*)" + key + R"((=|\s))", QRegularExpression::CaseInsensitiveOption);
		rule.format = format_known_key;
		rules_.append(rule);
	}

	/* comments */
	QTextCharFormat format_block_comment;
	format_block_comment.setForeground(colors::getQColor("syntax_comment"));
	rule.pattern = QRegularExpression(R"(^\s*[#;].*)");
	rule.format = format_block_comment;
	rules_.append(rule);
	rule.pattern = QRegularExpression(R"(([#;](?!$).*))");
	rules_.append(rule);

	/* equals sign */
	QTextCharFormat format_equal;
	format_equal.setForeground(Qt::black);
	rule.pattern = QRegularExpression(R"(=)");
	rule.format = format_equal;
	rules_.append(rule);
}

/**
 * @brief Send a chunk of text through the syntax highlighter.
 * @param[in] text The text to syntax-highlight.
 */
void SyntaxHighlighter::highlightBlock(const QString &text)
{
	for (const HighlightingRule &rule : qAsConst(rules_)) {
		QRegularExpressionMatchIterator mit( rule.pattern.globalMatch(text) );
		while (mit.hasNext()) { //run trough regex matches and set the stored formats
			QRegularExpressionMatch match = mit.next();
			setFormat(match.capturedStart(), match.capturedLength(), rule.format);
		}
	}
}

/**
 * @class PreviewWindow
 * @brief Default constructor for the PreviewWindow.
 * @details This constructor creates a tab bar to show multiple INI file versions in.
 * @param [in] parent The PreviewWindow's parent window (the main window).
 */
PreviewWindow::PreviewWindow(QMainWindow *parent) : QMainWindow(parent)
{
	this->setUnifiedTitleAndToolBarOnMac(true);
	file_tabs_ = new QTabWidget;
	connect(file_tabs_, &QTabWidget::tabCloseRequested, this, &PreviewWindow::closeTab);
	file_tabs_->setTabsClosable(true);
	this->setCentralWidget(file_tabs_);
	createMenu();
	createFindBar();

	//size this window (from "outside" via a timer to have it work on macOS; works without for the logger?):
	QTimer::singleShot(1, [=]{ dim::setDimensions(this, dim::PREVIEW); });
	this->setWindowTitle(tr("Preview") + " ~ " + QCoreApplication::applicationName());
}

/**
 * @brief Display the current INI file in a new tab.
 */
void PreviewWindow::addIniTab()
{
	/* get currently set INI values */
	QString ini_contents;
	QTextStream ss(&ini_contents);
	INIParser gui_ini = getMainWindow()->getIniCopy();
	getMainWindow()->getControlPanel()->setIniValuesFromGui(&gui_ini);

	/* text box for the current INI */
	auto *preview_editor( new QTextEdit );
	preview_editor->setStyleSheet("QTextEdit {background-color: " + colors::getQColor("syntax_background").name() +
	    "; color: " + colors::getQColor("syntax_invalid").name() + "}");
	highlighter_ = new SyntaxHighlighter(preview_editor->document());

	/* display the current GUI's contents */
	gui_ini.outputIni(ss);
	if (ini_contents.isEmpty()) {
		ini_contents = tr("#Empty INI file\n");
		previewStatus(tr("Open an application and load an INI file to view contents"));
	}
	preview_editor->setPlainText(ini_contents);
	QFileInfo file_info(getMainWindow()->getIni()->getFilename());
	QString file_name(file_info.fileName());
	QString file_path;
	if (file_info.exists()) //avoid warning
		file_path = file_info.absolutePath();
	if (file_name.isEmpty()) { //pick file name if no INI is opened yet
		file_name = "io_new.ini";
	} else {
		INIParser gui_ini = getMainWindow()->getIniCopy();
		(void) getMainWindow()->getControlPanel()->setIniValuesFromGui(&gui_ini);
		if (getMainWindow()->getIniCopy() != gui_ini)
			file_name += " *"; //asterisk for "not saved yet", unless it's only the info text
	}
	if (file_path.isEmpty())
		file_path = QDir::currentPath();
	file_tabs_->addTab(preview_editor, file_name);
	file_tabs_->setTabToolTip(file_tabs_->count() - 1, file_path);
	file_tabs_->setCurrentIndex(file_tabs_->count() - 1); //switch to new tab
	connect(preview_editor, &QTextEdit::textChanged, this, [=]{ textChanged(file_tabs_->count() - 1); });
	setMonospacedFont(preview_editor);
} //TODO: label for missing mandatory keys, tab completion for INI keys

/**
 * @brief Event listener for when the window is being closed.
 * @details This function allows the user to cancel the closing if there are unsaved
 * changes in the opened INI files.
 * @param[in] event The close event.
 */
void PreviewWindow::closeEvent(QCloseEvent *event)
{
	bool has_unsaved_changes = false; //run through tabs and check for asterisks
	for (int ii = 0; ii < file_tabs_->count(); ++ii) {
		if (file_tabs_->tabText(ii).endsWith("*")) {
			has_unsaved_changes = true;
			break;
		}
	}
	if (has_unsaved_changes &&
	    getSetting("user::inireader::warn_unsaved_ini", "value") == "TRUE") { //at least one tab has unsaved changes
		const int cancel = warnOnUnsavedIni();
		if (cancel == QMessageBox::Cancel) {
			event->ignore();
			return;
		}
	}
	event->accept();
}


/**
 * @brief Event listener for the ESC key.
 * @details Close the Preview on pressing ESC.
 * @param event The key press event that is received.
 */
void PreviewWindow::keyPressEvent(QKeyEvent *event)
{
	if (event->key() == Qt::Key_Escape)
		hideFindBar();
}

/**
 * @brief Close an INI file's tab.
 * @param[in] index Index of the tab that is being closed.
 */
void PreviewWindow::closeTab(int index)
{
	//Check for unsaved changes. Note that changes that cancel out leaving the INI file
	//unaltered will still trigger the warning (unlike the GUI).
	if (file_tabs_->tabText(index).endsWith("*") &&
	    getSetting("user::inireader::warn_unsaved_ini", "value") == "TRUE") { //not saved yet
		const int cancel = warnOnUnsavedIni();
		if (cancel == QMessageBox::Cancel)
			return;
	}

	file_tabs_->removeTab(index);
	if (file_tabs_->count() == 0)
		this->close();
}

/**
 * @brief Create the PreviewWindow's menu.
 */
void PreviewWindow::createMenu()
{
	/* File menu */
	QMenu *menu_file = this->menuBar()->addMenu(tr("&File"));
	auto *file_save = new QAction(getIcon("document-save"), tr("&Save"), menu_file);
	file_save->setShortcut( QKeySequence::Save );
	menu_file->addAction(file_save);
	connect(file_save, &QAction::triggered, this, &PreviewWindow::saveFile);
	auto *file_save_as = new QAction(getIcon("document-save-as"), tr("Save &as..."), menu_file);
	file_save_as->setShortcut( QKeySequence::SaveAs );
	menu_file->addAction(file_save_as);
	connect(file_save_as, &QAction::triggered, this, &PreviewWindow::saveFileAs);
	menu_file->addSeparator();
	auto *file_save_and_load = new QAction(tr("Save and load into GUI"), menu_file);
	menu_file->addAction(file_save_and_load);
	connect(file_save_and_load, &QAction::triggered, this, &PreviewWindow::saveFileAndLoadIntoGui);
	auto *file_load = new QAction(tr("Load into GUI"), menu_file);
	menu_file->addAction(file_load);
	connect(file_load, &QAction::triggered, this, &PreviewWindow::loadIntoGui);
	menu_file->addSeparator();
	auto *file_backup = new QAction(tr("Quicksave backup"), menu_file);
	menu_file->addAction(file_backup);
	connect(file_backup, &QAction::triggered, this, &PreviewWindow::quickBackup);

	/* Edit menu */
	QMenu *menu_edit = this->menuBar()->addMenu(tr("&Edit"));
	auto *edit_undo = new QAction(tr("Undo"), menu_edit);
	menu_edit->addAction(edit_undo);
	edit_undo->setShortcut(QKeySequence::Undo);
	connect(edit_undo, &QAction::triggered, this, [=]{ getCurrentEditor()->undo(); });
	auto *edit_redo = new QAction(tr("Redo"), menu_edit);
	menu_edit->addAction(edit_redo);
	edit_redo->setShortcut(QKeySequence::Redo);
	connect(edit_redo, &QAction::triggered, this, [=]{ getCurrentEditor()->redo(); });
	menu_edit->addSeparator();
	auto *edit_cut = new QAction(tr("Cut"), menu_edit);
	menu_edit->addAction(edit_cut);
	edit_cut->setShortcut(QKeySequence::Cut);
	connect(edit_cut, &QAction::triggered, this, [=]{ getCurrentEditor()->cut(); });
	auto *edit_copy = new QAction(tr("Copy"), menu_edit);
	menu_edit->addAction(edit_copy);
	edit_copy->setShortcut(QKeySequence::Copy);
	connect(edit_copy, &QAction::triggered, this, [=]{ getCurrentEditor()->copy(); });
	auto *edit_paste = new QAction(tr("Paste"), menu_edit);
	menu_edit->addAction(edit_paste);
	edit_paste->setShortcut(QKeySequence::Paste);
	connect(edit_paste, &QAction::triggered, this, [=]{ getCurrentEditor()->paste(); });
	auto *edit_select_all = new QAction(tr("Select all"), menu_edit);
	menu_edit->addAction(edit_select_all);
	edit_select_all->setShortcut(QKeySequence::SelectAll);
	connect(edit_select_all, &QAction::triggered, this, [=]{ getCurrentEditor()->selectAll(); });
	menu_edit->addSeparator();
	auto *edit_find = new QAction(getIcon("edit-find"), tr("&Find text..."), menu_edit);
	menu_edit->addAction(edit_find);
	edit_find->setShortcut(QKeySequence::Find);
	connect(edit_find, &QAction::triggered, this, &PreviewWindow::showFindBar);

	/* Transform menu */
	QMenu *menu_transform = this->menuBar()->addMenu(tr("&Transform"));
	auto *transform_whitespaces = new QMenu(tr("Whitespaces"), menu_transform);
	menu_transform->addMenu(transform_whitespaces);
	auto *transform_transform_singlews = new QAction(tr("To single spaces"), transform_whitespaces);
	connect(transform_transform_singlews, &QAction::triggered, this,
	    [=]{ onTransformMenuClick("transform_whitespace_singlews"); } );
	transform_whitespaces->addAction(transform_transform_singlews);
	auto *transform_transform_longestws = new QAction(tr("Adapt to longest keys"), transform_whitespaces);
	connect(transform_transform_longestws, &QAction::triggered, this,
	    [=]{ onTransformMenuClick("transform_whitespace_longestws"); } );
	transform_whitespaces->addAction(transform_transform_longestws);
	auto *transform_sort = new QMenu(tr("Sort"), menu_transform);
	menu_transform->addMenu(transform_sort);
	auto *transform_sort_alphabetically = new QAction(tr("Alphabetically"), transform_sort);
	connect(transform_sort_alphabetically, &QAction::triggered, this,
	    [=]{ onTransformMenuClick("transform_sort_alphabetically"); } );
	transform_sort->addAction(transform_sort_alphabetically);
	auto *transform_sort_order = new QAction(tr("In order of INI file"), transform_sort);
	connect(transform_sort_order, &QAction::triggered, this,
	    [=]{ onTransformMenuClick("transform_sort_order"); } );
	transform_sort->addAction(transform_sort_order);
	auto *transform_capitalization = new QMenu(tr("Capitalization"), menu_transform);
	menu_transform->addMenu(transform_capitalization);
	auto *transform_capitalization_upper = new QAction(tr("To upper case"), transform_capitalization);
	connect(transform_capitalization_upper, &QAction::triggered, this,
	    [=]{ onTransformMenuClick("transform_capitalization_upper"); } );
	transform_capitalization->addAction(transform_capitalization_upper);
	auto *transform_capitalization_lower = new QAction(tr("To lower case"), transform_capitalization);
	connect(transform_capitalization_lower, &QAction::triggered, this,
	    [=]{ onTransformMenuClick("transform_capitalization_lower"); } );
	transform_capitalization->addAction(transform_capitalization_lower);
	auto *transform_comments = new QMenu(tr("Comments"), menu_transform);
	menu_transform->addMenu(transform_comments);
	auto *transform_comments_delete = new QAction(tr("Delete comments"), transform_comments);
	connect(transform_comments_delete, &QAction::triggered, this,
	    [=]{ onTransformMenuClick("transform_comments_delete"); } );
	transform_comments->addAction(transform_comments_delete);
	auto *transform_reset = new QMenu(tr("Reset"), menu_transform);
	menu_transform->addMenu(transform_reset);
	auto *transform_reset_original = new QAction(tr("To original INI on file system"), transform_reset);
	connect(transform_reset_original, &QAction::triggered, this,
	    [=]{ onTransformMenuClick("transform_reset_original"); } );
	transform_reset->addAction(transform_reset_original);
	auto *transform_reset_full = new QAction(tr("To full INI"), transform_reset);
	connect(transform_reset_full, &QAction::triggered, this,
	    [=]{ onTransformMenuClick("transform_reset_full"); } );
	transform_reset->addAction(transform_reset_full);

	/* View menu */
	QMenu *menu_view = this->menuBar()->addMenu(tr("&View"));
	auto *view_close_tab = new QAction(tr("Close tab"), menu_view);
	view_close_tab->setShortcut(QKeySequence::Close);
	connect(view_close_tab, &QAction::triggered, this, [=]{ closeTab(file_tabs_->currentIndex()); });
	menu_view->addAction(view_close_tab);
}

/**
 * @brief Prepare a panel in the status bar that allows users to search for text in the preview.
 */
void PreviewWindow::createFindBar()
{
	find_text_ = new QLineEdit(this);
	connect(find_text_, &QLineEdit::textChanged, this, &PreviewWindow::onFindTextChanged);
	close_find_bar_ = new QToolButton(this);
	close_find_bar_->setIcon(getIcon("window-close"));
	close_find_bar_->setAutoRaise(true);
	connect(close_find_bar_, &QToolButton::clicked, this, [&]{ hideFindBar(); });
	this->statusBar()->addWidget(find_text_, 1);
	this->statusBar()->addWidget(close_find_bar_);
	hideFindBar();
}

/**
 * @brief Show the find bar.
 */
void PreviewWindow::showFindBar()
{
	previewStatus(QString( ));
	statusBar()->show();
	find_text_->show();
	close_find_bar_->show();
	find_text_->setFocus();
	find_text_->selectAll();
}

/**
 * @brief Hide the find bar.
 */
void PreviewWindow::hideFindBar()

{
	find_text_->hide();
	close_find_bar_->hide();
	statusBar()->hide();
}

/**
 * @brief Delegated event listener for when the preview text changes.
 * @details A lambda calls this function with the text box index to add an asterisk
 * to the title ("not saved").
 * @param[in] index Index of the tab the text changed in.
 */
void PreviewWindow::textChanged(const int &index)
{
	const QString file_name(file_tabs_->tabText(index));
	if (!file_name.endsWith("*"))
		file_tabs_->setTabText(index, file_name + " *");

}

/**
 * @brief Write an opened INI file to the file system.
 * @param[in] file_name The path to save to.
 */
void PreviewWindow::writeIniToFile(const QString &file_name)
{
	QFile outfile(file_name);
	if (!outfile.open(QIODevice::WriteOnly)) {
		previewStatus(tr("Could not open %1").arg(QDir::toNativeSeparators(file_name))); //temporary message
		return;
	}
	QTextStream ss(&outfile);
	const auto text_box( getCurrentEditor() );
	if (text_box)
		ss << text_box->toPlainText();
	outfile.close();

	const QFileInfo finfo(file_name); //switch the displayed name to new file (without asterisk)
	const QString shown_name(finfo.fileName());
	file_tabs_->setTabText(file_tabs_->currentIndex(), shown_name);
	previewStatus(tr("Saved to ") + file_name);

	setSetting("auto::history::last_preview_write", "path", finfo.absoluteDir().path());
}

/**
 * @brief Ask a user if they want to cancel an action.
 * @details This function is called before closing INI tabs with unsaved changes.
 * @return Button the user has clicked.
 */
int PreviewWindow::warnOnUnsavedIni()
{
	QMessageBox msgNotSaved;
	msgNotSaved.setWindowTitle("Warning ~ " + QCoreApplication::applicationName());
	msgNotSaved.setText(tr("<b>INI file not saved yet.</b>"));
	msgNotSaved.setInformativeText(tr("Your INI file(s) may contain unsaved changes."));
	msgNotSaved.setIcon(QMessageBox::Warning);
	msgNotSaved.setStandardButtons(QMessageBox::Cancel | QMessageBox::Discard);
	msgNotSaved.setDefaultButton(QMessageBox::Cancel);
	int clicked = msgNotSaved.exec();
	return clicked;
}

/**
 * @brief Show a temporary status message in the preview window. This text is volatile.
 * @param[in] text Text to show in the status.
 */
void PreviewWindow::previewStatus(const QString &text)
{
	statusBar()->showMessage(text);
	statusBar()->show(); //might be hidden from closing the find bar
	statusBar()->setToolTip(text);
}

/**
 * @brief Try to set a monospaced font for INIs that are space indented.
 * @param[in] editor Text editor to set monospaced font for.
 */
void PreviewWindow::setMonospacedFont(QTextEdit *editor)
{
	if (getSetting("user::appearance::preview_mono_font", "value") == "TRUE") {
		QFont mono_font( QFontDatabase::systemFont(QFontDatabase::FixedFont) ); //system recommendation
		mono_font.setPointSize(editor->font().pointSize());
		editor->setFont(mono_font);
	}
}

/**
 * @brief Helper function to get the current tab's text editor.
 * @return The text editor currently in view.
 */
QTextEdit * PreviewWindow::getCurrentEditor()
{
	return qobject_cast<QTextEdit *>(file_tabs_->currentWidget());
}

/**
 * @brief Helper function to retrieve the currently displayed file's name.
 * @return The current INI file's path.
 */
QString PreviewWindow::getCurrentFilename() const
{ //TODO: Store the filename properly without asterisk shuffling
	QString shown_name( file_tabs_->tabText(file_tabs_->currentIndex()) );
	if (shown_name.endsWith("*")) //remove "unsaved" asterisk for the file name
		shown_name.chop(2);
	return file_tabs_->tabToolTip(file_tabs_->currentIndex()) + "/" + shown_name;
}

/**
 * @brief Menu item to save an INI file to the file system.
 */
void PreviewWindow::saveFile()
{
	writeIniToFile(getCurrentFilename());
}

/**
 * @brief Menu item to save an INI file to the file system.
 */
void PreviewWindow::saveFileAs()
{
	QString start_path(getSetting("auto::history::last_preview_write", "path"));
	if (start_path.isEmpty())
		start_path = QDir::currentPath();
	QString proposed_name(file_tabs_->tabText(file_tabs_->currentIndex()));
	if (proposed_name.endsWith("*"))
		proposed_name.chop(2);
	const QString file_name = QFileDialog::getSaveFileName(this, tr("Save INI file"),
	    start_path + "/" + proposed_name, "INI files (*.ini *.INI);;All files (*)",
	    nullptr, QFileDialog::DontUseNativeDialog);
	if (file_name.isNull()) //cancelled
		return;
	writeIniToFile(file_name);
}

/**
 * @brief Menu item to save text to file and load into GUI.
 */
void PreviewWindow::saveFileAndLoadIntoGui()
{
	saveFile();
	getMainWindow()->openIni(getCurrentFilename());
}

/**
 * @brief Menu item to load into GUI.
 */
void PreviewWindow::loadIntoGui()
{
	INIParser preview_ini(getMainWindow()->getLogger());
	preview_ini.parseText(getCurrentEditor()->toPlainText());
	getMainWindow()->setGuiFromIni(preview_ini);
}

/**
 * @brief One-click action to save current contents to a new, unique file.
 */
void PreviewWindow::quickBackup()
{
	const QFileInfo finfo(getCurrentFilename());
	int counter = 1;
	const QString template_name( finfo.absoluteFilePath() + ".bak%1" );
	while (QFile::exists(template_name.arg(counter)))
		counter++;
	QFile outfile(template_name.arg(counter));
	if (!outfile.open(QIODevice::WriteOnly)) {
		previewStatus(tr("Could not open INI file for writing"));
		return;
	}
	QTextStream ss(&outfile);
	ss << getCurrentEditor()->toPlainText();
	outfile.close();
	previewStatus(tr("Saved to %1").arg(template_name.arg(counter)));
}

/**
 * @brief Event listener to text being changed in the find bar - go look for it.
 * @param[in] text Text the user wants to find.
 */
void PreviewWindow::onFindTextChanged(const QString &text)
{
	auto current_text_box_( getCurrentEditor() );
	QTextCursor cursor( current_text_box_->textCursor() );
	cursor.movePosition(QTextCursor::Start, QTextCursor::MoveAnchor, 1);
	current_text_box_->setTextCursor(cursor);
	const bool found = current_text_box_->find(text);
	if (found)
		find_text_->setStyleSheet(QString( ));
		//TODO: find next on Enter
	else
		find_text_->setStyleSheet("QLineEdit {color: " + colors::getQColor("warning").name() + "}");
}

/**
 * @brief Perform INI file transformations from the menu.
 * @details This function is called by a lambda with a menu action string.
 * @param[in] action String denoting the transformation to perform.
 */
void PreviewWindow::onTransformMenuClick(const QString &action)
{
	/*
	 * Currently these work on the original INI file that is loaded. This means that in order
	 * to do two transformations, the INI needs to be saved and loaded (menu), then transformed
	 * again.
	 * Instead of keeping a local INIParser copy, for now just show a note.
	 */

	QString ini_contents;
	QTextStream ss(&ini_contents);
	INIParser gui_ini = getMainWindow()->getIniCopy();
	getMainWindow()->getControlPanel()->setIniValuesFromGui(&gui_ini);

	if (action == "transform_whitespace_singlews") { //Transform all whitespaces to single spaces
		for (auto &sec : *gui_ini.getSections()) {
			for (auto &key : sec.getKeyValueList())
				sec.getKeyValue(key.first)->setKeyValWhitespaces(
				    std::vector<QString>( {"", " ", " ", " "} )); //(1)key(2)=(3)value(4)#comment
		}
		gui_ini.outputIni(ss);
	} else if (action == "transform_whitespace_longestws") { //look for longest key in section and use that many spaces
		for (auto &sec : *gui_ini.getSections()) {
			int max_key_length = 0;
			for (auto &key : sec.getKeyValueList()) {
				if (!key.second.getValue().isNull() && key.first.length() > max_key_length)
					max_key_length = key.first.length();
			}
			for (auto &key : sec.getKeyValueList()) {
				const int nr_ws = max_key_length - key.first.length() + 1;
				sec.getKeyValue(key.first)->setKeyValWhitespaces(
				    std::vector<QString>( {"", QString(" ").repeated(nr_ws), " ", " "} ) );
			}
		}
		gui_ini.outputIni(ss);
	} else if (action == "transform_sort_alphabetically") { //sort sections and within them the keys alphabetically
		gui_ini.outputIni(ss, true);
		//TODO: if a section is ordered before the default section, and the default section is not
		//output, then the keys switch section.
	} else if (action == "transform_sort_order") { //restore original INI
		gui_ini.outputIni(ss);
	} else if (action.startsWith("transform_capitalization_")) { //all sections, keys and values to upper/lower case
		//we do it through the INIParser so that comments stay the same
		const bool lower = (action == "transform_capitalization_lower");
		for (auto &sec : *gui_ini.getSections()) {
			sec.setName(lower? sec.getName().toLower() : sec.getName().toUpper());
			for (auto &key : sec.getKeyValueList()) {
				auto *keyvalue = sec.getKeyValue(key.first);
				keyvalue->setKey(lower? keyvalue->getKey().toLower() : keyvalue->getKey().toUpper());
				keyvalue->setValue(lower? keyvalue->getValue().toLower() : keyvalue->getValue().toUpper());
			}
		}
		gui_ini.outputIni(ss);
	} else if (action == "transform_comments_delete") { //delete all comments
		gui_ini.setBlockCommentAtEnd(QString());
		for (auto &sec : *gui_ini.getSections()) {
			sec.setBlockComment(QString());
			sec.setInlineComment(QString());
			for (auto &key : sec.getKeyValueList()) {
				auto *keyvalue = sec.getKeyValue(key.first);
				keyvalue->setBlockComment(QString());
				keyvalue->setInlineComment(QString());
			}
		}
		gui_ini.outputIni(ss);
	} else if (action == "transform_reset_original") { //reset to original INI
		getMainWindow()->getIni()->outputIni(ss);
	} else if (action == "transform_reset_full") { //reset to original INI plus GUI values
		gui_ini.outputIni(ss);
#ifdef DEBUG
	} else {
		qDebug() << "Signal mapping failed in PreviewWindow::onTransformMenuClick()";
#endif //def DEBUG
	}

	auto current_text_box_( getCurrentEditor() );
	current_text_box_->clear();
	current_text_box_->setText(ini_contents);
	previewStatus(tr("Note: Transformations work on the original loaded INI."));
}
