/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * Here are the panel classes, i. e. the GUI elements that are constructed from XML.
 * 2019-10
 */

#ifndef GUI_ELEMENTS_H
#define GUI_ELEMENTS_H

#include "src/main/constants.h"

#include <QCheckBox>
#include <QComboBox>
#include <QCryptographicHash>
#include <QGroupBox>
#include <QLabel>
#include <QLineEdit>
#include <QListWidget>
#include <QScrollArea>
#include <QString>
#include <QVBoxLayout>
#include <QWidget>
#include <QtXml>

#include <map>

#ifdef DEBUG
	#include <iostream>
#endif

QWidget * elementFactory(const QString &identifier, const QString &section, const QString &key,
    const QDomNode &options, const bool &no_spacers, QWidget *parent = nullptr);
void substituteKeys(QDomElement &parent_element, const QString &replace, const QString &replace_with);
QSpacerItem * buildSpacer();
void setLayoutMargins(QLayout *layout);
void addHelp(QHBoxLayout *layout, const QDomNode &options);

inline QString getQtKey(const QString &ini_key)
{
	/*
	 * Unfortunately, our arg1::arg2 syntax runs into Qt's sub-controls syntax, but the targeting
	 * with stylesheets only allows _ and - anyway so we hash the ID.
	 */
	QString hashed(QCryptographicHash::hash((ini_key.toLocal8Bit()), QCryptographicHash::Md5).toHex());
	//hashed.replace(QRegularExpression(R"([^\w-])"), "-"); //alternative
	return hashed;
}

class Atomic : public QWidget {
	Q_OBJECT

	public:
		Atomic(QString section, QString key, QWidget *parent = nullptr);
		QWidget * getPrimaryWidget() { return primary_widget_; }
		void setPrimaryWidget(QWidget *primary_widget);
		void setStyleMandatory(const bool &set = true);
		QString getId() const noexcept { return getQtKey(section_ + Cst::sep + key_); }

	protected:
		QString section_;
		QString key_;
		QWidget *primary_widget_ = nullptr; //widget to use for mandatory field highlighting

};

class Group : public Atomic {
	Q_OBJECT

	public:
		explicit Group(const QString &section, const QString &key, const bool &has_border = false,
		    const bool &grid_layout = false, const bool &is_frame = false, const QString &caption = QString(),
		    QWidget *parent = nullptr);
		void addWidget(QWidget *widget); //for vertical layout
		void addWidget(QWidget *widget, int row, int column, int rowSpan = 1, int columnSpan = 1,
		    Qt::Alignment alignment = Qt::Alignment()); //for grid layout
		QVBoxLayout * getLayout() const { return qobject_cast<QVBoxLayout *>(layout_); }
		QGridLayout * getGridLayout() const { return qobject_cast<QGridLayout *>(layout_); }
		void erase();
		int count() const;
		bool isEmpty() const;
		bool hasVisibleChildren() const;

	private:
		QGroupBox *box_ = nullptr;
		QLayout *layout_ = nullptr;
};

class Checklist : public Atomic {
	Q_OBJECT

	public:
		explicit Checklist(const QString &section, const QString &key, const QDomNode &options,
		    const bool &no_spacers, QWidget *parent = nullptr);

	private:
		bool setOptions(const QDomNode &options);
		QListWidget *list_ = nullptr;
		Group *container_ = nullptr;

	private slots:
		void listClick(QListWidgetItem *);

};

class Choice : public Atomic {
	Q_OBJECT

	public:
		explicit Choice(const QString &section, const QString &key, const QDomNode &options,
		    const bool &no_spacers, QWidget *parent = nullptr);

	private:
		bool setOptions(const QDomNode &options);
		QCheckBox *checkbox_ = nullptr;
		Group * checkbox_container_ = nullptr;
		Group * child_container_ = nullptr;

	private slots:
		void changedState(int index);
};

class Dropdown : public Atomic {
	Q_OBJECT

	public:
		explicit Dropdown(const QString &section, const QString &key, const QDomNode &options,
		    const bool &no_spacers, QWidget *parent = nullptr);
		Group * getContainer() const { return container_; }

	private:
		bool setOptions(const QDomNode &options);
		QComboBox *dropdown_ = nullptr;
		Group *container_ = nullptr;

	private slots:
		void itemChanged(int index);
};

class Replicator : public Atomic {
	Q_OBJECT

	public:
		explicit Replicator(const QString &section, const QString &key, const QDomNode &options,
		    const bool &no_spacers, QWidget *parent = nullptr);

	private:
		bool setOptions(const QDomNode &options);
		Group *container_ = nullptr;
		QDomNode templ_;
		int element_counter_ = 0;

	private slots:
		void replicate();
		void deleteLast();

};

class FilePath : public Atomic {
	Q_OBJECT

	public:
		explicit FilePath(const QString &section, const QString &key, const QDomNode &options,
		    const bool &no_spacers, QWidget *parent = nullptr);

	private:
		bool setOptions(const QDomNode &options);
		QLineEdit *path_text_ = nullptr;
		bool path_only_ = false;
		QString extensions_ = "";

	private slots:
		void openFile();
};

class GridPanel : public Atomic {
	Q_OBJECT

	public:
		explicit GridPanel(const QString &section, const QString &key, const QDomNode &options,
		    QWidget *parent = nullptr);

	private:
		bool setOptions(const QDomNode &options);
		QGridLayout *grid_layout_ = nullptr;
};

class Helptext : public QLabel {
	Q_OBJECT

	public:
		explicit Helptext(const QString &text, const bool &single_line, QWidget *parent = nullptr);

	private:
		int getMinTextSize(const QString &text, const int &standard_width);
		QLabel *help_ = nullptr;
};

class HorizontalPanel : public Atomic {
	Q_OBJECT

	public:
		explicit HorizontalPanel(const QString &section, const QString &key, const QDomNode &options,
		    const bool &no_spacers, QWidget *parent = nullptr);

	private:
		bool setOptions(const QDomNode &options, const bool &no_spacers);
		QHBoxLayout *horizontal_layout_ = nullptr;
};

class Label : public Atomic {
	Q_OBJECT

	public:
		explicit Label(const QString &section, const QString &key, const QDomNode &options,
		    const bool &no_spacers, QWidget *parent = nullptr);

	private:
		int getColumnWidth(const QString &text, const int& min_width);
		QLabel *label_ = nullptr;

};

class Number : public Atomic {
	Q_OBJECT

	public:
		explicit Number(const QString &section, const QString &key, const QDomNode &options,
		    const bool &no_spacers, QWidget *parent = nullptr);

	private:
		bool setOptions(const QDomNode &options);
		QAbstractSpinBox *number_element_ = nullptr;
};

class Selector : public Atomic {
	Q_OBJECT

	public:
		explicit Selector(const QString &section, const QString &key, const QDomNode &options,
		    const bool &no_spacers, QWidget *parent = nullptr);

	private:
		bool setOptions(const QDomNode &options);
		QComboBox *dropdown_ = nullptr;
		Group *container_ = nullptr;
		QDomNode templ_;
		std::map<QString, Group *> container_map_;

	private slots:
		void addPanel();
		void removePanel();
};

class Textfield : public Atomic {
	Q_OBJECT

	public:
		explicit Textfield(const QString &section, const QString &key, const QDomNode &options,
		    const bool &no_spacers, QWidget *parent = nullptr);

	private:
		QLineEdit *textfield_ = nullptr;
};

#endif //GUI_ELEMENTS_H
