/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "gui_elements.h"
#include "src/main/colors.h"
#include "src/main/dimensions.h"
#include "src/main/inishell.h"
#include "src/main/XMLReader.h"

#include <QApplication>
#include <QDoubleSpinBox>
#include <QFileDialog>
#include <QHBoxLayout>
#include <QPalette>
#include <QPushButton>
#include <QSpinBox>

#include <climits> //for the number panel limits
#include <utility> //for modern constructor move semantics

#ifdef DEBUG
	#include <iostream>
#endif

/**
 * @brief Object factory for the panels.
 * @details Construct an object from a string name (often read from an XML).
 * @param[in] identifier Name of the object.
 * @param[in] section INI section the controlled values belong to.
 * @param[in] key INI key of the controlled value.
 * @param[in] options The current XML node with all options and children.
 * @param[in] no_spacers If available, set a tight layout for the object.
 * @param[in] parent The parent widget.
 * @return An object of the panel family to manipulate values.
 */
QWidget * elementFactory(const QString &identifier, const QString &section, const QString &key,
    const QDomNode &options, const bool& no_spacers, QWidget *parent)
{
	if (options.toElement().attribute("replicate") == "true") {
		return new Replicator(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "alternative") {
		return new Dropdown(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "checklist") {
		return new Checklist(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "choice") {
		return new Choice(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "file" || identifier.toLower() == "path") {
		return new FilePath(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "grid") {
		return new GridPanel(section, key, options, parent);
	} else if (identifier.toLower() == "label") {
		return new Label(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "number") {
		return new Number(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "selector") {
		return new Selector(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "text") {
		return new Textfield(section, key, options, no_spacers, parent);
	} else if (identifier.toLower() == "horizontal") {
		return new HorizontalPanel(section, key, options, no_spacers, parent);
	} else {
		topLog(QApplication::tr("Unknown parameter object in XML file: \"") + identifier +
		    "\"", colors::getQColor("error"));
		topStatus(QApplication::tr("Unknown XML parameter type"));
		return nullptr; //will throw a Qt warning
	}
}

/**
 * @brief Substitute values in keys and texts.
 * @details This is for child elements that inherit a property from its parent which should be
 * displayed (e. g. "TA" in "TA::FILTER1 = ..."). The function traverses through the whole child
 * tree recursively.
 * @param[in] parent_element Parenting XML node holding the desired values.
 * @param[in] replace String to replace.
 * @param[in] replace_with Text to replace the string with.
 */
void substituteKeys(QDomElement &parent_element, const QString &replace, const QString &replace_with)
{
	for (QDomElement element = parent_element.firstChildElement(); !element.isNull(); element = element.nextSiblingElement()) {
		QString key(element.attribute("key"));
		element.setAttribute("key", key.replace(replace, replace_with));
		QString text(element.attribute("caption"));
		element.setAttribute("caption", text.replace(replace, replace_with)); //for labels
		substituteKeys(element, replace, replace_with);
	}
}

/**
 * @brief Build a standard spacer item for widget positioning.
 * @details This is the only little workaround we use for our layouts. The scroll areas are allowed
 * to resize everything, and sometimes they do so aggressively; for example, they ignore left-aligned
 * elements and move them to the right if the screen gets very big. If on the other hand we try to
 * adjust the sizes of all child widgets manually via .adjustSize() update(), etc., then we run into
 * troubles that require far uglier hacks if they can be solved at all. It has proven best to try not
 * to meddle with the layout manager.
 * The solution here is to add a huge spacer to a list of left-aligned widgets if we want to keep all of
 * them a fixed (small) size.
 * @return Spacer element that can be added to a layout.
 */
QSpacerItem * buildSpacer()
{
	return new QSpacerItem(getMainWindow()->width() * 2, 0, QSizePolicy::Maximum); //huge spacer
}

/**
 * @brief Set margins of a layout.
 * @details This controls how much space there is between widgets, i. e. the spacing in the main GUI.
 * @param [in] layout Any layout.
 */
void setLayoutMargins(QLayout *layout)
{
	//                  left, top, right, bottom
	layout->setContentsMargins(2, 1, 2, 1); //with our nested groups we want to keep them tight
}

/**
 * @brief Convenience call to add a Helptext object to the end of a vertical layout.
 * @param[in] layout Layout to add the Helptext to.
 * @param[in] options Parent XML node controlling the appearance of the help text.
 */
void addHelp(QHBoxLayout *layout, const QDomNode &options)
{
	QDomElement help_element(options.firstChildElement("help")); //dedicated <help> tag if there is one
	bool single_line = (help_element.attribute("nowrap") == "true"); //single line extending the scroll bar
	QString helptext(help_element.text());
	if (helptext.isEmpty()) //check if there is a help property, e. g. <parameter type=... help="helptext">
		helptext = options.toElement().attribute("help");
	if (!helptext.isEmpty()) {
		auto *help = new Helptext(helptext, single_line);
		layout->addWidget(help, 0, Qt::AlignRight);
	}
}

////////////////////////////////////////
///        ATOMIC base class         ///
////////////////////////////////////////

/**
 * @class Atomic
 * @brief Base class of most panels.
 * @details The panels inherit from this class which provides some common functionality
 * like handling the object IDs.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this panel.
 * @param[in] parent The parent widget.
 */
Atomic::Atomic(QString section, QString key, QWidget *parent)
    : QWidget(parent), section_(std::move(section)), key_(std::move(key))
{
	//do nothing
}

/**
 * @brief Set a panel's primary widget.
 * @details The widget pointed to this way is responsible for actually changing a value and
 * is used for highlighting purposes.
 * @param primary_widget The panel's main widget.
 */
void Atomic::setPrimaryWidget(QWidget *primary_widget)
{
	primary_widget_ = primary_widget;
	primary_widget_->setObjectName(getQtKey(getId()));
}

/**
 * @brief Set a property indicating that the value this panel controls is mandatory.
 * @param set Set style on/off.
 */
void Atomic::setStyleMandatory(const bool &set)
{
	this->primary_widget_->setProperty("mandatory", set);
}

////////////////////////////////////////
///         GROUPING element         ///
////////////////////////////////////////

/**
 * @class Group
 * @brief Default Group constructor
 * @details Group is the central grouping element which the GUI building recursion works on.
 * Essentially it is a wrapper for QGroupBox.
 * The main program tab holds Groups in which the panels go. If the panels feature
 * child elements themselves they too own a group in which they are put.
 * @param[in] section Name of the INI section the group corresponds to. This could be used
 * to style the groups differently depending on the section, but for now it is unused. This
 * parameter does not affect the childrens' settings.
 * @param[in] key (INI) key for the group. Does not affect the childrens' keys.
 * @param[in] has_border Optional border around the group.
 * @param[in] grid_layout Elements are not placed vertically (default) but in a grid layout.
 * @param[in] is_frame Act as a frame with border and title.
 * @param[in] caption If the group is a frame the caption is displayed as title.
 * @param[in] parent The parent widget.
 */
Group::Group(const QString &section, const QString &key, const bool &has_border,
	    const bool &grid_layout, const bool &is_frame, const QString &caption, QWidget *parent)
	    : Atomic(section, key, parent)
{
	box_ = new QGroupBox(is_frame? caption : QString()); //all children go here, only frame can have a title
	setPrimaryWidget(box_);
	if (grid_layout)
		layout_ = new QGridLayout; //mainly for the grid panel
	else
		layout_ = new QVBoxLayout; //frame is always this
	box_->setLayout(layout_); //we can add widgets at any time

	if (!is_frame) { //normal group
		if (!has_border) {
			box_->setStyleSheet("QGroupBox#" + getQtKey(getId()) + " {border: none; margin-top: 0px}");
		} else {
			box_->setStyleSheet("QGroupBox#" + getQtKey(getId()) +" {border: 1px solid " +
			    colors::getQColor("groupborder").name() + "; border-radius: 6px;}");
			setLayoutMargins(layout_);
		}
	} else { //it's a frame
		box_->setStyleSheet(" \
			    QGroupBox#" + getQtKey(getId()) + "  {font: bold; border: 2px solid " +
			    colors::getQColor("frameborder").name() + "; border-radius: 6px; margin-top: 8px; color: " +
			    colors::getQColor("frametitle").name() + ";} QGroupBox::title#" + getQtKey(getId()) +
			    " {subcontrol-origin: margin; left: 17px; padding: 0px 5px 0px 5px; \
			}");
		box_->layout()->setContentsMargins(Cst::frame_horizontal_margins, Cst::frame_vertical_margins,
		    Cst::frame_horizontal_margins, Cst::frame_vertical_margins); //a little room for the frame
	}

	auto *main_layout = new QVBoxLayout; //layout for the derived class to assign the box_ to it
	setLayoutMargins(main_layout); //tighter placements
	main_layout->addWidget(box_, 0, Qt::AlignTop); //alignment important when we hide/show children
	this->setLayout(main_layout);
}

/**
 * @brief Add a child widget to the group's vertical layout.
 * @param[in] widget The widget to add.
 */
void Group::addWidget(QWidget *widget)
{
	auto *layout = qobject_cast<QVBoxLayout *>(layout_);
	layout->addWidget(widget, 0, Qt::AlignTop); //alignment needed after we hide/show groups
}

/**
 * @brief Add a child widget to the group's grid layout.
 * @details This is called by the GridPanel to add widgets in a raster.
 * @param[in] widget The widget to add.
 * @param[in] row Row position of the widget (starts at 0).
 * @param[in] column Column position of the widget (starts at 0).
 * @param[in] rowSpan The widget spans this many rows (default: 1).
 * @param[in] columnSpan The widget spans this many columns (default: 1).
 * @param[in] alignment Alignment of the widget within the raster point.
 */
void Group::addWidget(QWidget *widget, int row, int column, int rowSpan, int columnSpan,
    Qt::Alignment alignment)
{
	auto *layout = qobject_cast<QGridLayout *>(layout_); //TODO: safechecks
	layout->addWidget(widget, row, column, rowSpan, columnSpan, alignment);
}

/**
 * @brief Delete all child panels of the group.
 * @details The group itself is not deleted, but the container is rendered useless.
 */
void Group::erase()
{
	setUpdatesEnabled(false); //disable painting in case we have a lot of content
	delete box_;
	setUpdatesEnabled(true);
}

/**
 * @brief Retrieve the number of child panels.
 * @return The number of child panels.
 */
int Group::count() const
{
	return (this->getLayout()->count());
}

/**
 * @brief Check if the group has child panels.
 * @return True if there is at least one child.
 */
bool Group::isEmpty() const
{
	return (this->count() == 0);
}

/**
 * @brief Check if the group has visible child groups.
 * @details If a child group is visible but has no visible content, it is still considered visible.
 * This is enough for Checklist and Choice which hide the groups sufficiently.
 * @return True if at least one child group is visible.
 */
bool Group::hasVisibleChildren() const
{
	QList<Group *> list = box_->findChildren<Group *>();
	for (auto &widget : list) {
		if (widget->isVisible())
			return true;
	}
	return false;
}

////////////////////////////////////////
///         CHECKLIST panel          ///
////////////////////////////////////////

/**
 * @class Checklist
 * @brief Default constructor for a Checklist panel.
 * @details A Checklist shows a checkable list where each list item can have arbitrary children
 * which are displayed below if the item is checked and hidden if it's not.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this Checklist.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Checklist::Checklist(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	/* label and list */
	auto *key_label = new Label(section_, key_, options, no_spacers);
	list_ = new QListWidget;
	setPrimaryWidget(list_);
	connect(list_, &QListWidget::itemClicked, this, &Checklist::listClick);

	/* layout of the basic elements */
	auto *checklist_layout = new QHBoxLayout;
	setLayoutMargins(checklist_layout);
	checklist_layout->addWidget(key_label, 0, Qt::AlignTop);
	checklist_layout->addWidget(list_);

	/* layout of basic elements plus children */
	container_ = new Group(section, "_checklist_container"); //generic name that could be used for stylesheets
	container_->setVisible(false); //container for children activates only when clicked (saves space)
	auto *layout = new QVBoxLayout;
	setLayoutMargins(layout);
	layout->addLayout(checklist_layout);
	layout->addWidget(container_);
	this->setLayout(layout);

	setOptions(options); //fill list items
}

/**
 * @brief Parse options for a Checklist from XML.
 * @param[in] options XML node holding the Checklist.
 * @return True if all options were set successfully.
 */
bool Checklist::setOptions(const QDomNode &options)
{
	for (QDomElement op = options.firstChildElement("option"); !op.isNull(); op = op.nextSiblingElement("option")) {
		//add the option value as list item:
		auto *item = new QListWidgetItem(op.attribute("value"), list_);
		item->setFlags(item->flags() | Qt::ItemIsUserCheckable);
		item->setCheckState(Qt::Unchecked);

		//construct the children of this option:
		auto *item_group = new Group(section_, "_checklist_itemgroup"); //group all elements of this option together
		recursiveBuild(op, item_group, section_);
		container_->addWidget(item_group);
		item_group->setVisible(false); //becomes visible when checked
	}
	//TODO: set size according to number of items if below threshold
	//TODO: style the scrollbar (buttons only when small)
	//this many rows should be visible (with some padding):
	list_->setFixedHeight(list_->sizeHintForRow(0) * Cst::nr_items_visible + Cst::safety_padding);
	return true;
}

/**
 * @brief Event handler for clicks in the list widget.
 * @details This handles showing and hiding children belonging to list items.
 * @param[in] item Item that was clicked.
 */
void Checklist::listClick(QListWidgetItem *item)
{
	if (item->checkState() == Qt::Checked) //click in line is enough to check/uncheck
		item->setCheckState(Qt::Unchecked);
	else
		item->setCheckState(Qt::Checked);

	//show/hide children of list option that has been clicked:
	setUpdatesEnabled(false);
	QVBoxLayout *group_layout = container_->getLayout();
	auto *item_group = qobject_cast<Group*>(group_layout->itemAt(list_->currentRow())->widget());
	item_group->setVisible(item->checkState() == Qt::Checked && !item_group->isEmpty());

	//do not display container when empty (a few blank pixels):
	container_->setVisible(true); //else, all children are invisible
	container_->setVisible(container_->hasVisibleChildren());
	setUpdatesEnabled(true); //calls update()
}

////////////////////////////////////////
///           CHOICE panel           ///
////////////////////////////////////////

/**
 * @class Choice
 * @brief Default constructor for a Choice panel.
 * @details A choice panel shows a list of checkboxes, each of which controls showing/hiding of
 * additional options.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this Choice panel.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Choice::Choice(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	const QDomElement element(options.toElement());

	//grouping element of the list of checkboxes:
	checkbox_container_ = new Group(section, "_choice_checkbox_container", false, true); //grid layout
	setPrimaryWidget(checkbox_container_);
	//grouping element for all children:
	child_container_ = new Group(section, "_choice_child_container"); //vertical layout
	child_container_->setVisible(false);
	auto *key_label = new Label(section_, key_, options, no_spacers);
	checkbox_container_->addWidget(key_label, 0, 0, 1, 1);

	/* layout for checkboxes and children together */
	auto *layout = new QVBoxLayout;
	setLayoutMargins(layout);
	layout->addWidget(checkbox_container_);
	layout->addWidget(child_container_);
	this->setLayout(layout);

	setOptions(options); //build children
}

/**
 * @brief Parse options for a Choice panel from XML.
 * @param[in] options XML node holding the Choice panel.
 * @return True if all options were set successfully.
 */
bool Choice::setOptions(const QDomNode &options)
{
	int counter = 0;
	for (QDomElement op = options.firstChildElement("option"); !op.isNull(); op = op.nextSiblingElement("option")) {
		auto *checkbox = new QCheckBox(op.attribute("value"));
		//connect to lambda function to emit current index (modern style signal mapping):
		connect(checkbox, &QCheckBox::stateChanged, this, [=] { changedState(counter); });
		checkbox_container_->addWidget(checkbox, counter, 1);

		/* help text */
		QString helptext = op.firstChildElement("help").text();
		if (helptext.isEmpty()) //same as addHelp but for a certain grid position
			helptext = op.attribute("help");
		auto *help = new Helptext(helptext, false);
		checkbox_container_->addWidget(help, counter, 2, 1, 1, Qt::AlignRight);

		/* child elements of this checkbox */
		auto *item_group = new Group(section_, "_item_choice");
		recursiveBuild(op, item_group, section_);
		item_group->setVisible(false);
		child_container_->addWidget(item_group);
		counter++;
	}
	return true;
}

/**
 * @brief Event listener for when a single checkbox is checked/unchecked.
 * @details This function shows/hides child elements when a checkbox changes.
 * @param[in] index The index/row of the clicked item.
 */
void Choice::changedState(int index)
{
	QVBoxLayout *group_layout = child_container_->getLayout(); //get item number 'index' from the child group's layout
	auto *item_group = qobject_cast<Group *>(group_layout->itemAt(index)->widget());
	auto *clicked_box =
	    qobject_cast<QCheckBox *>(checkbox_container_->getGridLayout()->itemAtPosition(index, 1)->widget());
	setUpdatesEnabled(false);
	//show the clicked child's group on item check if it's not empty:
	item_group->setVisible(clicked_box->checkState() == Qt::Checked && !item_group->isEmpty());
	child_container_->setVisible(true);
	child_container_->setVisible(child_container_->hasVisibleChildren());
	setUpdatesEnabled(true);

}

////////////////////////////////////////
///          DROPDOWN panel          ///
////////////////////////////////////////

/**
 * @class Dropdown
 * @brief Default constructor for a Dropdown panel.
 * @details A Dropdown panel shows a dropdown menu where additional options pop up depending on the
 * selected menu item.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this Choice panel.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Dropdown::Dropdown(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	/* label and combo box */
	dropdown_ = new QComboBox(this);
	setPrimaryWidget(dropdown_);
	dropdown_->setMinimumWidth(Cst::tiny); //no tiny elements
	connect(dropdown_, QOverload<int>::of(&QComboBox::currentIndexChanged), this, &Dropdown::itemChanged);

	/* layout of the basic elements */
	auto *dropdown_layout = new QHBoxLayout;
	setLayoutMargins(dropdown_layout);
	if (!key.isEmpty()) {
		auto *key_label = new Label(section_, key_, options, no_spacers);
		dropdown_layout->addWidget(key_label, 0, Qt::AlignLeft);
	}
	dropdown_layout->addWidget(dropdown_, 0, Qt::AlignLeft);
	//add a big enough spacer to the right to keep the buttons to the left (unless it's a horizontal layout):
	if (!no_spacers)
		dropdown_layout->addSpacerItem(buildSpacer());
	addHelp(dropdown_layout, options);

	/* layout of the basic elements plus children */
	container_ = new Group(section, "_dropdown", true);
	auto *layout = new QVBoxLayout(this);
	setLayoutMargins(layout);
	layout->addLayout(dropdown_layout);
	layout->addWidget(container_);
	this->setLayout(layout);

	setOptions(options); //construct child panels
	emit itemChanged(0); //select first option
}

/**
 * @brief Parse options for a Dropdown panel from XML.
 * @param[in] options XML node holding the Dropdown panel.
 * @return True if all options were set successfully.
 */
bool Dropdown::setOptions(const QDomNode &options)
{
	for (QDomElement op = options.firstChildElement("option"); !op.isNull(); op = op.nextSiblingElement("option")) {
		/* add childrens' names to dropdown and build them */
		dropdown_->addItem(op.attribute("value"));
		auto *item_group = new Group(section_, "_item_dropdown"); //group all elements of this option together
		recursiveBuild(op, item_group, section_);
		container_->addWidget(item_group);
	}
	return true;
}

/**
 * @brief Event listener for when a dropdown menu item is selected.
 * @details This function shows/hides the children corresponding to the selected dropdown item.
 * @param[in] index Index/number of the selected item.
 */
void Dropdown::itemChanged(int index)
{
	QVBoxLayout *group_layout = container_->getLayout();
	for (int ii = 0; ii < group_layout->count(); ++ii) {
		auto *group_item = qobject_cast<Group *>(group_layout->itemAt(ii)->widget());
		group_item->setVisible((ii == index));
		if (ii == index) //hide the container if the displayed group is empty
			container_->setVisible(!group_item->isEmpty());
	}
}

////////////////////////////////////////
///         REPLICATOR panel         ///
////////////////////////////////////////

/**
 * @class Replicator
 * @brief Default constructor for a Replicator.
 * @details A Replicator holds a widget which it can replicate with the click of a button.
 * It does not have a separate identifier but rather it is activated in any given panel with the "replicate"
 * attribute. The number of the created panel is propagated to all children via "#".
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this Replicator.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Replicator::Replicator(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	container_ = new Group(QString(), "_replicator", true);

	/* label, dropdown menu and buttons */
	auto *key_label = new Label(section_, key_, options, no_spacers);
	auto *plus_button = new QPushButton("+");
	auto *minus_button = new QPushButton("-");
	plus_button->setFixedSize(Cst::std_button_width, Cst::std_button_height);
	minus_button->setFixedSize(plus_button->size());
	connect(plus_button, &QPushButton::clicked, this, &Replicator::replicate); //replicate child
	connect(minus_button, &QPushButton::clicked, this, &Replicator::deleteLast); //delete last child

	/* layout of the basic elements */
	auto *replicator_layout = new QHBoxLayout;
	setLayoutMargins(replicator_layout);
	replicator_layout->addWidget(key_label);
	replicator_layout->addWidget(plus_button, 0, Qt::AlignLeft);
	replicator_layout->addWidget(minus_button, 0, Qt::AlignLeft);
	if (!no_spacers)
		replicator_layout->addSpacerItem(buildSpacer()); //keep widgets to the left
	addHelp(replicator_layout, options);

	/* layout of the basic elements plus children */
	auto *layout = new QVBoxLayout;
	setLayoutMargins(layout);
	layout->addLayout(replicator_layout);
	layout->addWidget(container_);
	this->setLayout(layout);

	setOptions(options); //set the child
	container_->setVisible(false); //only visible when an item is selected
}

/**
 * @brief Parse options for a Replicator from XML.
 * @param[in] options XML node holding the Replicator.
 * @return True if all options were set successfully.
 */
bool Replicator::setOptions(const QDomNode &options)
{
	templ_ = options; //save a reference to the child XML node (shallow copy)
	return true;
}

/**
 * @brief Event listener for the plus button: replicate the child widget.
 * @details The child was saved as XML node, here this is parsed and built.
 */
void Replicator::replicate()
{
	element_counter_++;
	QDomNode node = prependParent(templ_); //prepend artificial parent node for recursion (runs through children)
	node.firstChildElement().setAttribute("replicate", "false"); //set node to normal element to be constructed
	QDomElement element = node.toElement();

	//recursively inject the element's number into the childrens' keys:
	substituteKeys(element, "#", QString::number(element_counter_));

	setUpdatesEnabled(false);
	Group *new_group = new Group(section_, "_replicator_item");
	recursiveBuild(node, new_group, section_); //construct the children
	container_->addWidget(new_group);
	container_->setVisible(true);
	setUpdatesEnabled(true);
}

/**
 * @brief Event listener for the minus button: Remove the instance of the child widget created last.
 */
void Replicator::deleteLast()
{
	if (element_counter_ == 0)
		return; //no more widgets left
	if (element_counter_ == 1)
		container_->setVisible(false);
	auto *to_delete = qobject_cast<Group *>(container_->getLayout()->takeAt(element_counter_ - 1)->widget());
	if (to_delete) {
		to_delete->erase(); //delete the group's children
		delete to_delete; //delete the group itself
	}
	element_counter_--;
}

////////////////////////////////////////
///         FILE/PATH panel          ///
////////////////////////////////////////

/**
 * @class FilePath
 * @brief Default constructor for a file/path picker.
 * @details The file/path picker displays a dialog to select either a file or a folder.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this file/path picker.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
FilePath::FilePath(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	/* label, text field and "open" button */
	auto *key_label = new Label(section_, key_, options, no_spacers);
	path_text_ = new QLineEdit; //textfield with one line
	setPrimaryWidget(path_text_);
	auto *open_button = new QPushButton("...");
	open_button->setFixedSize(Cst::medium_button_width, Cst::std_button_height);
	connect(open_button, &QPushButton::clicked, this, &FilePath::openFile);

	/* layout of the basic elements */
	auto *filepath_layout = new QHBoxLayout;
	setLayoutMargins(filepath_layout);
	filepath_layout->addWidget(key_label, 0, Qt::AlignLeft);
	filepath_layout->addWidget(path_text_);
	filepath_layout->addWidget(open_button, 0, Qt::AlignLeft);
	addHelp(filepath_layout, options);
	this->setLayout(filepath_layout);

	setOptions(options); //file or path
}

/**
 * @brief Parse options for a file/patch picker from XML.
 * @param[in] options XML node holding the file/path picker.
 * @return True if all options were set successfully.
 */
bool FilePath::setOptions(const QDomNode &options)
{
	if (options.toElement().attribute("type") == "path")
		path_only_ = true;
	for (QDomElement op = options.firstChildElement("option"); !op.isNull(); op = op.nextSiblingElement("option")) {
		const QString ext = op.attribute("extension"); //selectable file extensions can be set
		extensions_ += ext + (ext.isEmpty()? "" : ";;");
	}
	if (extensions_.isEmpty())
		extensions_= "All Files (*)";
	else
		extensions_.chop(2); //remove trailing ;;
	return true;
}

/**
 * @brief Event listener for the open button.
 * @details Open a file or path by displaying a dialog window.
 */
void FilePath::openFile()
{
	QString path;
	if (path_only_) { //TODO: remember last location
		path = QFileDialog::getExistingDirectory(this, tr("Open Folder"), "./",
		    QFileDialog::DontUseNativeDialog | QFileDialog::ShowDirsOnly);
	} else {
		path = QFileDialog::getOpenFileName(this, tr("Open File"), "./",
		    extensions_, nullptr, QFileDialog::DontUseNativeDialog);
	}
	path_text_->setText(path);
}

////////////////////////////////////////
///           GRID raster            ///
////////////////////////////////////////

/**
 * @class GridPanel
 * @brief Default constructor for a GridPanel.
 * @details A GridPanel is a simple grid layout that organizes child widgets on a raster. The children
 * are given enclosed in <option> tags in the XML file.
 * @param[in] section The INI section is set for style targeting.
 * @param[in] key INI key is used for an optional label and ignored otherwise.
 * @param[in] options XML node responsible for this panel with all desired children.
 * @param[in] parent The parent widget.
 */
GridPanel::GridPanel(const QString &section, const QString &key, const QDomNode &options, QWidget *parent)
    : Atomic(section, key, parent)
{
	grid_layout_ = new QGridLayout(this); //only holds a grid layout for child panels
	setLayoutMargins(grid_layout_);
	this->setLayout(grid_layout_);

	setOptions(options); //construct children
}

/**
 * @brief Parse options for a GridPanel from XML.
 * @param[in] options XML node holding the GridPanel.
 * @return True if all options were set successfully.
 */
bool GridPanel::setOptions(const QDomNode &options)
{
	if (!key_.isEmpty()) {
		auto *key_label = new Label(section_, key_, options, true);
		grid_layout_->addWidget(key_label, 0, 0, Qt::AlignVCenter | Qt::AlignLeft);
	}
	//construct all child elements:
	for (QDomElement op = options.firstChildElement("option"); !op.isNull(); op = op.nextSiblingElement("option")) {
		auto *item_group = new Group(section_, "_grid_itemgroup");
		recursiveBuild(op, item_group, section_, true); //recursive build with horizontal space savings
		const int row = op.attribute("row").toInt() - 1; //indices start at 1 in XML file
		const int column = op.attribute("column").toInt() - 1; //TODO: safechecks
		grid_layout_->addWidget(item_group, row, column, Qt::AlignVCenter | Qt::AlignLeft);
	}
	return true;
}

////////////////////////////////////////
///         HELPTEXT element         ///
////////////////////////////////////////

/**
 * @class Helptext
 * @brief Default constructor for a Helptext.
 * @details A Helptext is a styled label with properties set to display it on the far right of a layout.
 * The help text can either be displayed with a fixed width for a uniform look.
 * Or it can be displayed in a single line, in which case it is shown right next to the panel it describes
 * and extens as far as the text goes, viewable with the main horizontal scroll bar.
 * @param[in] text The help text.
 * @param[in] single_line If set, do not try to align the text in multiple columns. Rather, display it
 * in a single line and use the main scroll bars for it.
 * @param[in] parent The parent widget.
 */
Helptext::Helptext(const QString &text, const bool &single_line, QWidget *parent) : QLabel(parent)
{
	this->setTextFormat(Qt::RichText);
	this->setTextInteractionFlags(Qt::TextBrowserInteraction);
	this->setOpenExternalLinks(true); //clickable links
	QFont font(this->font());
	font.setPointSize(Cst::help_font_size); //smaller font
	this->setFont(font);
	QPalette label_palette(this->palette()); //custom color
	label_palette.setColor(QPalette::WindowText, colors::getQColor("helptext"));
	this->setPalette(label_palette);

	if (!single_line) { //no space to preceding elements, users scroll to the end
		this->setWordWrap(true); //multi line
		const QFontMetrics font_metrics(font);
		const int text_width = font_metrics.horizontalAdvance(text); //real size of text on screen
		//Texts that fit in a single line are right-aligned so they sit at the very right rather than
		//on the left side of a box that is help_width big (thus creating margins):
		if (text_width <= Cst::help_width)
			this->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
		//standard Helptext width, or the text width if it is smaller than that:
		this->setFixedWidth(getMinTextSize(text, Cst::help_width));
	}
	this->setText(text);
}
/**
 * @brief Compare the standard Helptext width with the width of the text.
 * @details If the width of the text is smaller than the standard help text width then this
 * function returns the smaller text width. This way the element floats freely without
 * unnecessary margins.
 * @param[in] text Text that will be displayed.
 * @param[in] min_width
 * @return The fixed width to use for this Helptext.
 */
int Helptext::getMinTextSize(const QString &text, const int &standard_width)
{
	const QFontMetrics font_metrics(this->font());
	const int text_width = font_metrics.horizontalAdvance(text); //horizontal pixels
	if (text_width < standard_width)
		return text_width + 1; //tiny bit of room to not wrap unexpectedly
	else
		return standard_width;
}

////////////////////////////////////////
///        HORIZONTAL raster         ///
////////////////////////////////////////

/**
 * @class HorizontalPanel
 * @brief Default constructor for a HorizontalPanel.
 * @details A HorizontalPanel is a simple layout that organizes child widgets horizontally. The children
 * are given enclosed in <option> tags in the XML file.
 * @param[in] section The INI section is set for style targeting.
 * @param[in] key INI key is used for an optional label and ignored otherwise.
 * @param[in] options XML node responsible for this panel with all desired children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
HorizontalPanel::HorizontalPanel(const QString &section, const QString &key, const QDomNode &options,
    const bool &no_spacers, QWidget *parent) : Atomic(section, key, parent)
{
	horizontal_layout_ = new QHBoxLayout(this); //only holds a horizontal layout for child panels
	setLayoutMargins(horizontal_layout_);
	this->setLayout(horizontal_layout_);

	setOptions(options, no_spacers); //construct children
	addHelp(horizontal_layout_, options); //children and the panel can both have help texts
}

/**
 * @brief Parse options for a HorizontalPanel from XML.
 * @param[in] options XML node holding the HorizontalPanel.
 * @return True if all options were set successfully.
 */
bool HorizontalPanel::setOptions(const QDomNode &options, const bool &no_spacers)
{
	if (!key_.isEmpty()) { //display caption
		auto *key_label = new Label(section_, key_, options, true);
		horizontal_layout_->addWidget(key_label, 0, Qt::AlignLeft | Qt::AlignCenter);
	}
	/* build all children */
	for (QDomElement op = options.firstChildElement("option"); !op.isNull(); op = op.nextSiblingElement("option")) {
		auto *item_group = new Group(section_, "_horizontal_itemgroup");
		recursiveBuild(op, item_group, section_, true); //recursive build with horizontal space savings
		horizontal_layout_->addWidget(item_group, 0, Qt::AlignVCenter | Qt::AlignLeft);
	}
	if (!no_spacers)
		horizontal_layout_->addSpacerItem(buildSpacer()); //keep widgets to the left
	return true;
}

////////////////////////////////////////
///          LABEL element           ///
////////////////////////////////////////

/**
 * @class Label
 * @brief Default constructor for a Label.
 * @details A label displays styled text, for example for INI keys.
 * @param[in] section INI section the described value belongs to. Could be used for style targeting
 * but is ignored otherwise.
 * @param[in] key INI key corresponding to the value that is being described. If no caption is set then
 * this text will be displayed.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Label::Label(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	/* label styling */
	QString caption = options.toElement().attribute("caption"); //TODO: allow to set text as option alternatively
	if (caption.isEmpty())
		caption = key;

	label_ = new QLabel(caption);
	setPrimaryWidget(label_);
	QPalette label_palette(this->palette());
	label_palette.setColor(QPalette::WindowText, colors::getQColor("key"));
	label_->setPalette(label_palette);

	/* main layout */
	auto *layout = new QVBoxLayout;
	layout->addWidget(label_);
	setLayoutMargins(layout);
	this->setLayout(layout);
	if (!no_spacers)
		this->setMinimumWidth(getColumnWidth(caption, Cst::label_width)); //set to minimal required room
	//set a fixed with so that window resizing does not introduce unneccessary margins:
	this->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
}

/**
 * @brief Retrieve the minimal width required for the label.
 * @details A fixed minimal width is set for labels so that they act as the first "column" of the interface
 * making it more visually appealing. If a label needs more space than that it is extended.
 * @param[in] text Text that will be displayed.
 * @param[in] min_width The minimum "column" width for the (INI key) labels.
 * @return The size to set the label to.
 */
int Label::getColumnWidth(const QString &text, const int& min_width)
{
	const QFontMetrics font_metrics(this->font());
	const int text_width = font_metrics.horizontalAdvance(text);
	if (text_width > min_width)
		return text_width + Cst::label_padding;
	else
		return min_width + Cst::label_padding;
}

////////////////////////////////////////
///           NUMBER panel           ///
////////////////////////////////////////

/**
 * @class Number
 * @brief Default constructor for a Number panel.
 * @details A number panel displays and manipulates a float or integer value.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this Number panel.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Number::Number(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	/* number widget depending on which type of number to display */
	const QString format = options.toElement().attribute("format");
	if (format == "float" || format.isEmpty()) {
		number_element_ = new QDoubleSpinBox;
	} else if (format == "integer" || format == "integer+") {
		number_element_ = new QSpinBox;
	} else if (format == "expression") {
		//not implemented yet
	} else {
		topLog(tr("Unknown number format in XML file"), colors::getQColor("warning"));
	}
	setPrimaryWidget(number_element_);
	auto *key_label = new Label(section_, key_, options, no_spacers);
	number_element_->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
	number_element_->setMinimumWidth(Cst::tiny);

	/* layout of basic elements */
	auto *number_layout = new QHBoxLayout;
	setLayoutMargins(number_layout);
	number_layout->addWidget(key_label);
	number_layout->addWidget(number_element_);
	if (!no_spacers)
		number_layout->addSpacerItem(buildSpacer()); //keep widgets to the left
	addHelp(number_layout, options);

	/* main layout */
	auto *layout = new QVBoxLayout(this);
	setLayoutMargins(layout);
	layout->addLayout(number_layout);
	this->setLayout(layout);

	setOptions(options); //min, max, default, ...
}

/**
 * @brief Parse options for a Number panel from XML.
 * @param[in] options XML node holding the Number panel.
 * @return True if all options were set successfully.
 */
bool Number::setOptions(const QDomNode &options)
{
	const QString format = options.toElement().attribute("format");
	const QString maximum = options.toElement().attribute("maximum");
	const QString minimum = options.toElement().attribute("minimum");
	const QString unit = options.toElement().attribute("unit");
	const QString def_value = options.toElement().attribute("default");

	if (format == "float") {
		auto *spinbox = qobject_cast<QDoubleSpinBox *>(number_element_);
		bool success = true;
		/* minimum and maximum, choose whole range if they aren't set */
		double min = minimum.isEmpty()? std::numeric_limits<double>::min() : minimum.toDouble(&success);
		if (!success)
			topLog(tr("Could not parse minimum value for key ") + key_);
		double max = maximum.isEmpty()? std::numeric_limits<double>::max() : maximum.toDouble(&success);
		if (!success)
			topLog(tr("Could not parse maximum value for key ") + key_);
		/* unit and default value */
		spinbox->setRange(min, max);
		if (!unit.isEmpty()) //for the space before the unit
			spinbox->setSuffix(" " + unit);
		if (!def_value.isEmpty()) {
			double def = def_value.toDouble(&success);
			if (!success)
				topLog(tr("Could not parse default value for key ") + key_);
			if (def < min) //defaults can not exceed the limits
				def = min;
			if (def > max)
				def = max;
			spinbox->setValue(def);
		}
	} else if (format == "integer" || format == "integer+") {
		auto spinbox = qobject_cast<QSpinBox *>(number_element_);
		if (options.toElement().attribute("wrap") == "true") //circular wrapping when min/max is reached
			spinbox->setWrapping(true);
		/* minimum and maximum */
		bool success = true;
		int min = 0; //integer+
		if (format == "integer")
			min = minimum.isEmpty()? std::numeric_limits<int>::min() : minimum.toInt(&success);
		if (!success)
			topLog(tr("Could not parse minimum value for key ") + key_);
		int max = maximum.isEmpty()? std::numeric_limits<int>::max() : maximum.toInt(&success);
		if (!success)
			topLog(tr("Could not parse maximum value for key ") + key_);
		spinbox->setRange(min, max);
		/* unit and default value */
		if (!unit.isEmpty())
			spinbox->setSuffix(" " + unit);
		if (!def_value.isEmpty()) {
			int def = def_value.toInt(&success);
			if (!success)
				topLog(tr("Could not parse default value for key ") + key_);
			if (def < min)
				def = min;
			if (def > max)
				def = max;
			spinbox->setValue(def);
		}
	} //endif format
	return true; //TODO: some checks
}

////////////////////////////////////////
///          SELECTOR panel          ///
////////////////////////////////////////

/**
 * @class Selector
 * @brief Default constructor for a Selector.
 * @details A selector panel allows to select from a list of text pieces (e. g. meteo parameters), either from
 * a fixed dropdown list or with possible free text input. Its single child panel must be declared as "template"
 * in its attributes. With the click of a button the child element is duplicated, inheriting the text piece.
 * @param[in] section INI section the controlled value belongs to.
 * @param[in] key INI key corresponding to the value that is being controlled by this Selector.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Selector::Selector(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	/* label, dropdown menu and buttons */
	auto *key_label = new Label(section_, key_, options, no_spacers);
	dropdown_ = new QComboBox;
	setPrimaryWidget(dropdown_);
	dropdown_->setMinimumWidth(Cst::tiny); //no tiny elements
	dropdown_->setEditable(true); //free text with autocomplete
	dropdown_->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed); //size of biggest item
	auto *plus_button = new QPushButton("+"); //TODO: icons
	auto *minus_button = new QPushButton("-");
	plus_button->setFixedSize(dropdown_->height(), dropdown_->height()); //buttons have same height as dropdown
	minus_button->setFixedSize(plus_button->size());
	connect(plus_button, &QPushButton::clicked, this, &Selector::addPanel);
	connect(minus_button, &QPushButton::clicked, this, &Selector::removePanel);

	/* layout of the basic elements */
	auto *selector_layout = new QHBoxLayout;
	setLayoutMargins(selector_layout);
	selector_layout->addWidget(key_label);
	selector_layout->addWidget(dropdown_, 0, Qt::AlignLeft);
	selector_layout->addWidget(plus_button, 0, Qt::AlignLeft);
	selector_layout->addWidget(minus_button, 0, Qt::AlignLeft);
	if (!no_spacers)
		selector_layout->addSpacerItem(buildSpacer()); //keep buttons from wandering to the right

	/* layout for the basic elements plus children */
	container_ = new Group(QString(), "_selector", true);
	container_->setVisible(false); //only visible with items
	auto *layout = new QVBoxLayout(this);
	setLayoutMargins(layout);
	layout->addLayout(selector_layout);
	layout->addWidget(container_);
	this->setLayout(layout);

	setOptions(options); //construct children
}

/**
 * @brief Parse options for a Selector from XML.
 * @param[in] options XML node holding the Selector.
 * @return True if all options were set successfully.
 */
bool Selector::setOptions(const QDomNode &options)
{
	//TODO: optionally no free text
	for (QDomElement par = options.firstChildElement("parameter"); !par.isNull(); par = par.nextSiblingElement("parameter")) {
		if (par.attribute("template") == "true") {
			templ_ = par; //save the node describing the child (shallow copy)
			break; //TODO: check for bad syntax (multiple templates)
		}
	}
	for (QDomElement op = options.firstChildElement("option"); !op.isNull(); op = op.nextSiblingElement("option"))
		dropdown_->addItem(op.attribute("value")); //fill list of texts the selector shows as default
	return true;
}

/**
 * @brief Event listener for the plus button: add a child panel to the selector.
 * @details This function replicates the child panel from XML and passes the selected text.
 */
void Selector::addPanel()
{
	const QString drop_text(dropdown_->currentText());
	if (drop_text.isEmpty()) {
		topStatus(tr("Empty field in dropdown text"));
		return;
	}
	if (container_map_.count(drop_text) != 0) { //only one panel per piece of text
		topStatus(tr("Item already exists"));
		return;
	}
	topStatus(""); //above messages could be confusing if still displayed from recent click

	//we clone the child node (deep copy for string replacement) and put it in a dummy parent:
	QDomNode node(prependParent(templ_)); //nest for recursion
	QDomElement element(node.toElement());
	substituteKeys(element, "%", dropdown_->currentText()); //recursive substitution for all children
	 //draw it on next call (don't use as template again):
	node.firstChildElement("parameter").setAttribute("template", "false");

	/* construct all children and grandchildren */
	Group *new_group = new Group(section_, "_panel");
	recursiveBuild(node, new_group, section_);
	container_->addWidget(new_group);
	container_->setVisible(true);

	const auto panel_pair = std::make_pair(drop_text, new_group);
	container_map_.insert(panel_pair); //keep an index of text to panel number
}

/**
 * @brief Event listener for the minus button: remove a panel for the selected text piece.
 * @details If some text is present in the dropdown menu, the corresponding child is deleted.
 */
void Selector::removePanel()
{
	const QString drop_text(dropdown_->currentText());
	auto it = container_map_.find(drop_text); //look up if item exists in map
	if (it != container_map_.end()) {
		topStatus(""); //no "does not exist" error message from earlier
		it->second->erase(); //delete the group's children
		delete it->second; //delete the group itself
		container_map_.erase(it);
		if (container_map_.empty()) //no more children - save a couple of blank pixels
			container_->setVisible(false);
	} else {
		topStatus(tr("Item does not exist"));
	}
}

////////////////////////////////////////
///          TEXT panel              ///
////////////////////////////////////////

/**
 * @class Textfield
 * @brief Default constructor for a Textfield.
 * @details A Textfield is used to enter plain text.
 * @param[in] key INI key corresponding to the value that is being controlled by this Textfield.
 * @param[in] options XML node responsible for this panel with all options and children.
 * @param[in] no_spacers Keep a tight layout for this panel.
 * @param[in] parent The parent widget.
 */
Textfield::Textfield(const QString &section, const QString &key, const QDomNode &options, const bool &no_spacers,
    QWidget *parent) : Atomic(section, key, parent)
{
	/* label and text box */
	auto *key_label = new Label(section_, key_, options, no_spacers);
	textfield_ = new QLineEdit; //single line text box
	setPrimaryWidget(textfield_);
	auto *textfield_layout = new QHBoxLayout;
	setLayoutMargins(textfield_layout);
	textfield_layout->addWidget(key_label, 0, Qt::AlignLeft);
	textfield_layout->addWidget(textfield_, 0);

	/* choose size of text box */
	const QString size(options.toElement().attribute("size"));
	if (size.toLower() == "medium")
		textfield_->setMinimumWidth(Cst::medium_textbox_width);
	else
		textfield_->setMinimumWidth(Cst::tiny);
	if (!no_spacers && size.toLower() != "large") //"large": text box extends to window width
		textfield_layout->addSpacerItem(buildSpacer());

	addHelp(textfield_layout, options);
	this->setLayout(textfield_layout);
}
