/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "inishell.h"
#include "inishell.h"

#ifdef DEBUG
	#include <iostream>
#endif

/**
 * @brief Recursively build the interface.
 * @details This function traverses the XML tree and builds input/output panels into a grouping element.
 * At the top level, this grouping element is held by the scroll panels (one per tab). Panels are
 * added there, and these panels can themselves own groups, which again can be built in.
 * @param[in] parent_node The parent XML node. This corresponds to the parent node/group and
 * therefore holds all options for the parent Group, and all info about children that should be built.
 * @param[in] parent_group The Group to build in.
 * @param[in] parent_section The current section. If omitted, the parent section is chosen.
 * @param[in] no_spacers The parent group requests to save space and build a tight layout.
 */
void recursiveBuild(const QDomNode &parent_node, Group *parent_group, const QString &parent_section, const bool &no_spacers)
{
	const QString default_section(Cst::default_section); //use this if no section is given

	getMainWindow()->setUpdatesEnabled(false); //disable painting until done
	for (QDomNode current_node = parent_node.firstChild(); !current_node.isNull(); current_node = current_node.nextSibling()) {
		/* read some attributes of the current node */
		QDomElement element = current_node.toElement();
		const QString current_section(element.firstChildElement("section").attribute("name"));
		QString section(current_section);
		if (current_section.isEmpty())
			section = parent_section.isEmpty()? default_section : parent_section;
		const QString key(element.attribute("key"));
		Group *group_to_add_to(parent_group);
		if (group_to_add_to == nullptr) //we are at top level -> find or if necessary create tab
			group_to_add_to = getMainWindow()->getControlPanel()->getTabScroll(section)->getGroup();

		if (element.tagName() == "frame") { //visual grouping by a frame with title
			const QString frametitle(element.attribute("caption"));
			Group *frame = new Group(current_section, key, true, false, true, frametitle);
			group_to_add_to->addWidget(frame);
			recursiveBuild(current_node, frame, section); //all children go into the frame
		} else if (element.tagName() == "parameter") { //a panel
			if (element.attribute("template") == "true") //Selector element will handle this
				continue;
			/* build the desired object, add it to the parent group, and recursively build it children */
			QWidget *new_element = elementFactory(element.attribute("type"), section, key,
			    current_node, no_spacers);
			group_to_add_to->addWidget(new_element);
			recursiveBuild(current_node, group_to_add_to, section, no_spacers);
		} //endif element type
	} //endfor current_node
	getMainWindow()->setUpdatesEnabled(true);
}
