/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * The top level INI file interface.
 * 2019-10
 */

#ifndef INIPARSER_H
#define INIPARSER_H

#include "constants.h"
#include "src/gui/Logger.h"

#include <QCoreApplication> //for translations
#include <QTextStream>

#include <map>
#include <vector>

using cfloat = double; //where applicable, use this type for floating point values

/**
 * @class Key
 * @brief A container for keys found in INI files.
 * @details This class holds the string "KEY" from "KEY = VALUE" in an INI file. It is an object that
 * saves all properties that are extracted for this string.
 */
class Key {
	public:
		Key(const size_t &max_parts) : partial_keys_(max_parts) {}
		Key(const QString &in_fullkey, const std::vector<QString> &in_parts)
		{
			full_key_ = in_fullkey;
			partial_keys_ = in_parts;
		}
		//comparison operator to function as key in a map:
		bool operator< (const Key &in_key) const { return this->full_key_ < in_key.getFullKey(); }

		QString getFullKey() const {return full_key_; }
		void setFullKey(const QString &in_key) { full_key_ = in_key; }
		QString getPartialKey(const size_t &keynr) const { return partial_keys_.at(keynr); }
		bool setPartialKey(const size_t &keynr, const QString &in_part)
		{
			if (partial_keys_.at(keynr) == in_part)
				return false; //nothing changed
			partial_keys_.at(keynr) = in_part;
			rebuildFullKey();
			return true;
		}
		size_t countParts() const { return partial_keys_.size(); }

	private:
		void rebuildFullKey()
		{
			full_key_.clear();
			for (QString &key : partial_keys_) {
				if (!key.isEmpty())
					full_key_ += key;
			}
		}
		QString full_key_ = ""; //the full key (e. g. param::option1::arg1)
		std::vector<QString> partial_keys_; //key parts separated by our separator (e. g. "::")
};

/**
 * @class Value
 * @brief A container for values found in INI files.
 * @details This class holds the string "VALUE" from "KEY = VALUE" in an INI file. It is an object that
 * saves all properties that are extracted for this string.
 */
class Value {
	public:
		void setValue(const QString &in_value) { val_ = in_value; }
		QString getValue() const { return val_; }
		QString getInlineComment() const { return inline_comment_; }
		void setInlineComment(const QString &in_comment) { inline_comment_ = in_comment; }
		QString getBlockComment() const { return block_comment_; }
		void setBlockComment(const QString &in_comment) { block_comment_ = in_comment; }

	private:
		QString val_ = "";
		QString inline_comment_ = "";
		QString block_comment_ = "";
};

/**
 * @class Section
 * @brief A container for sections found in INI files.
 * @details This class holds a section name "[SECTION]" from an INI file. It is an object that
 * saves all properties that are extracted for this string.
 */
class Section {
	public:
		//Comparison operator to be able to be used as a map key. It compares the section
		//name as can be expected:
		bool operator< (const Section &in_section) const { return this->name_ < in_section.getName(); }
		QString getName() const { return name_; } //the section name
		void setName(const QString &in_name) { name_ = in_name; }
		QString getComment() const { return inline_comment_; } //inline comment (e. g. "KEY = VAL #COMMENT")
		void setComment(const QString &in_comment) { inline_comment_ = in_comment; }
		QString getBlockComment() const { return block_comment_; } //block comment preceeding the section
		void setBlockComment(const QString &in_comment) { block_comment_ = in_comment; }
		void print(QTextStream &ss) const { //print section with comments to stream
			ss << block_comment_;
			if (!name_.isEmpty())
				ss << Cst::section_open << name_ << Cst::section_close << endl;
			if (!inline_comment_.isEmpty())
				ss << " " << inline_comment_;
		}

	private:
		QString name_ = "";
		QString inline_comment_ = "";
		QString block_comment_ = "";
};

/**
 * @class KeyValues
 * @brief Holds a collection of key/value-pairs
 * @details This class houses a number of key/value-pairs that can be grouped in, for example,
 * a Section. It wraps a map in a way that its contents can be accessed as usual via
 * KeyValues[Key key], but also as KeyValues[int key] which standard containers can't do both at the
 * same time. This gives rise to the possibility of both range based loops (alphabetical access) and for loops
 * (access in insertion order).
 */
class KeyValues {
	public:
		KeyValues() = default;
		Value &operator[](const Key &key) { return kv_pairs_[key]; } //propagate access by Key to main map
		//access in ordering of insertion: map[int key]
		std::map<Key, Value>::iterator &operator[](size_t index) { return ordered_key_values_.at(index); }
		std::map<Key, Value>::iterator begin() { return kv_pairs_.begin(); } //enable range loops
		std::map<Key, Value>::iterator end() { return kv_pairs_.end(); } //use map iterator
		size_t size() const { return kv_pairs_.size(); }
		void set(const Key &key, const Value &value) { //insert or change a Key/Value combination
			auto pair = std::make_pair(key, value);
			std::pair<std::map<Key, Value>::iterator, bool> it_insert;
			it_insert = kv_pairs_.insert(pair);
			if (it_insert.second) //new insertion
				ordered_key_values_.push_back(it_insert.first); //TODO: on file input, check for doubled keys
		}
		void clear() {
			kv_pairs_.clear();
			ordered_key_values_.clear();
		}

	private:
		std::map<Key, Value> kv_pairs_; //main map key to value
		std::vector<std::map<Key, Value>::iterator> ordered_key_values_; //stores insertion order
};

/**
 * @class SectionList
 * @brief A collection of sections found in an INI file.
 * @details This class stores all [SECTIONS] that are parsed from an INI file.
 * Like KeyValues, easy access is given via SectionList[Key key] , but additionally index based
 * access offers Sections in order of insertion via SectionList[int key].
 */
class SectionList {
	public:
		KeyValues &operator[](const Section &section) { return section_list_[section]; } //propagate access by Section
		Section &operator[](size_t index) { return ordered_sections_.at(index); } //access by index
		std::map<Section, KeyValues>::iterator begin() { return section_list_.begin(); } //enable range loops
		std::map<Section, KeyValues>::iterator end() { return section_list_.end(); }
		size_t size() const { return section_list_.size(); }
		//change or insert a key/value pair into a section:
		void set(const Section &section, const Key &key, const Value &value) {
			if (section_list_.find(section) == section_list_.end())
				ordered_sections_.push_back(section);
			section_list_[section].set(key, value); //set in the main KeyValues list
		}
		void clear() {
			section_list_.clear();
			ordered_sections_.clear();
		}

	private:
		std::map<Section, KeyValues> section_list_; //main map
		std::vector<Section> ordered_sections_; //stores insertion order TODO: can we store the adress?
};

class INIParser {
	public:
		INIParser(Logger &in_logger);
		INIParser(const QString &in_file, Logger &in_logger);
		void setFilename(const QString &in_filename) noexcept;
		bool setFile(const QString &in_file); //read and parse
		bool parseFresh();
		bool outputIni(QTextStream &out_ss, const bool &alphabetical = false);
		void printKeyValuePair(const std::pair<Key, Value> &keyval, QTextStream &ss);
		bool writeIni();

	private:
		bool evaluateComment(const QString &line, QString &out_comment); //is line a comment?
		bool evaluateSection(const QString &line, Section &out_section);
		bool evaluateKeyValue(const QString &line, Key &key, Value &out_value);

		QString filename_ = "";
		SectionList ini_sections_;

		Logger &logger_instance_; //the main logger is kept so we don't have to constantly grab it
};

#endif //INIPARSER_H
