/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

#include "INIParser.h"
#include "inishell.h"
#include "src/gui/MainWindow.h"
#include "src/main/colors.h"
#include "src/main/Error.h"

#include <QTextStream> //for endl

INIParser::INIParser(Logger &in_logger) : logger_instance_(in_logger)
{
	//do nothing
}

INIParser::INIParser(const QString &in_file, Logger &in_logger) : logger_instance_(in_logger)
{
	setFile(in_file);
}

void INIParser::setFilename(const QString &in_filename) noexcept
{
	filename_ = in_filename;
}

bool INIParser::setFile(const QString &in_file)
{
	setFilename(in_file);
	return parseFresh();
}

bool INIParser::parseFresh()
{
	ini_sections_.clear();
	QFile infile(filename_);
	if (!infile.open(QIODevice::ReadOnly | QIODevice::Text)) {
		Error(QCoreApplication::tr("Could not open INI file for reading"), "",
		    filename_ + ":\n" + infile.errorString());
		return false;
	}
	QString current_comment;
	Section current_section;
	size_t linecount = 0;

	QTextStream tstream(&infile);
	while (!tstream.atEnd()) {
		QString line = tstream.readLine();
		linecount++;
		QString comment;
		const bool comment_success = evaluateComment(line, comment);
		if (comment_success) {
			current_comment += comment + "\n";
			continue;
		}
		const bool section_success = evaluateSection(line, current_section);
		if (section_success) {
			current_section.setBlockComment(current_comment);
			current_comment.clear();
		} else {
			static const size_t key_max_parts = 3; //hardcoded for now: arg1::arg2::arg3
			Key key(key_max_parts);
			Value value;
			const bool key_value_success = evaluateKeyValue(line, key, value);
			if (key_value_success) {
				value.setBlockComment(current_comment);
				current_comment.clear();
				ini_sections_.set(current_section, key, value);
			} else {
				QString msg = QString(QApplication::tr("Undefined format on line %1 of file \"%2\"")).arg(linecount).arg(filename_)
				    + ": " + line;
				logger_instance_.log(msg, colors::getQColor("warning"));
				topStatus(QApplication::tr("Invalid line in file \"") + filename_ + "\"");
			}
		}
	}
	return true;
}

bool INIParser::outputIni(QTextStream &out_ss, const bool& alphabetical)
{
	if (alphabetical) { //range loop: use container iterators (sorted)
		for (auto &it : ini_sections_) { //run through sections
			const Section section = it.first;
			section.print(out_ss);

			KeyValues keyvals = it.second;
			for (auto &kv : keyvals) {
				printKeyValuePair(kv, out_ss);
				out_ss << endl;
			} //endfor kv
		} //endfor it
	} else { //for loop: use order of insertion (unsorted)
		for (size_t ii = 0; ii < ini_sections_.size(); ++ii) {
			const Section section = ini_sections_[ii];
			section.print(out_ss);

			KeyValues keyvals = ini_sections_[section];
			for (size_t jj = 0; jj < keyvals.size(); ++jj) {
				auto keyval = std::make_pair(keyvals[jj]->first, keyvals[jj]->second);
				printKeyValuePair(keyval, out_ss);
				out_ss << endl;
			} //endfor jj
		} //endfor ii
	} //endif alphabetical

	return true;
}

void INIParser::printKeyValuePair(const std::pair<Key, Value> &keyval, QTextStream &ss)
{
	ss << keyval.second.getBlockComment();
	ss << keyval.first.getFullKey() << " = " << keyval.second.getValue(); //TODO: respect users' whitespaces
	if (!keyval.second.getInlineComment().isEmpty())
		ss << " " << keyval.second.getInlineComment();
}

bool INIParser::writeIni()
{
	QTextStream ss;
	const bool output_success = outputIni(ss);
	return output_success;
}

bool INIParser::evaluateComment(const QString &line, QString &out_comment)
{
	if (line.isEmpty())
		return  true; //reproduce empty lines too
	static const QString regex_comment( R"(^\s*[#;].*)" );
	/*                                          |
	 *                                         (1)
	 * (1) Any line that starts with ; or #, optionally prefaced by whitespaces
	 */
	static const size_t idx_comment = 0;

	static const QRegularExpression rex(regex_comment);
	const QRegularExpressionMatch matches = rex.match(line);
	out_comment = matches.captured(idx_comment);
	return matches.hasMatch();
}

bool INIParser::evaluateSection(const QString &line, Section &out_section)
{
	static const QString regex_section( R"(\[(\w+)\]\s*([#;].*)*)" );
	/*                                           |          |
	 *                                          (1)        (2)
	 * (1) Alphanumeric string (can't be empty) enclosed by brackets
	 * (2) Comment started with ; or #
	 */
	static const size_t idx_total = 0;
	static const size_t idx_name = 1;
	static const size_t idx_comment = 2;

	static const QRegularExpression rex(regex_section);
	const QRegularExpressionMatch matches = rex.match(line);
	const bool total_match = (line == matches.captured(idx_total));

	if (total_match) {
		out_section.setName(matches.captured(idx_name));
		out_section.setComment(matches.captured(idx_comment));
		return true;
	}
	return  false;
}

bool INIParser::evaluateKeyValue(const QString &line, Key &key, Value &out_value)
{

	static const QString regex_keyval(
	    R"((\w+)(::\w+){0,1}(::\w+){0,1}(\s*)=(\s*)(;$|#$|.+?)(\s*)(#.*|;.*|$))" );
	/*       |        \       /           \    /        \             |
	 *      (1)        \ (2) /             \  /         (4)          (5)
	 *                                      (3)
	 * (1) Alphanumeric string for the key (can't be empty)
	 * (2) Either 0, 1 or 2 times an appended ::argument (wrote like this to get both back as matches)
	 * (3) Any number of whitespacescaround =
	 * (4) Either a single ; or # (e. g. csv delimiter) or any non-empty string for the value
	 * (5) The value is either ended by a comment or the end of the line
	 */
	static const size_t idx_total = 0;
	static const size_t idx_first_partial = 1;
	static const size_t idx_value = 6;
	static const size_t idx_comment = 8;

	static const QRegularExpression rex(regex_keyval);
	const QRegularExpressionMatch matches = rex.match(line);

	const bool total_match = (line == matches.captured(idx_total));

	if (total_match) {
		for (size_t ii = 0; ii < key.countParts(); ++ii)
			key.setPartialKey(ii, matches.captured(static_cast<int>(idx_first_partial + ii)));
		out_value.setValue(matches.captured(idx_value));
		out_value.setInlineComment(matches.captured(idx_comment));
		return true;
	}
	return  false;
}
