/*****************************************************************************/
/*  Copyright 2019 WSL Institute for Snow and Avalanche Research  SLF-DAVOS  */
/*****************************************************************************/
/* This file is part of INIshell.
   INIshell is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   INIshell is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with INIshell. If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * The main window that is displayed on startup.
 * 2019-10
 */

#include "MainWindow.h"
#include "src/main/colors.h"
#include "src/main/Error.h"
#include "src/main/dimensions.h"
#include "src/main/INIParser.h"
#include "src/main/inishell.h"
#include "src/main/XMLReader.h"
#include "src/gui/Settings.h"

#include <QApplication>
#include <QGroupBox>
#include <QMenuBar>
#include <QStatusBar>
#include <QString>
#include <QSysInfo>
#include <QToolBar>

#ifdef DEBUG
	#include <iostream>
	#include <sstream>
#endif

/**
 * @class MainWindow
 * @brief Constructor for the main window.
 * @param[in] xml_settings Settings for the static INIshell GUI.
 * @param[in] parent Unused since the main window is kept in scope by the entry point function.
 */
MainWindow::MainWindow(QDomDocument &xml_settings, QMainWindow *parent)
    : QMainWindow(parent), logger_(this), xml_settings_(xml_settings)
{
	logger_.logSystemInfo();

	/* retrieve and set main window geometry */
	dim::ScreenSize screen;
	dim::MainDimension dim_inishell(screen);
	dim::setSize(this, dim_inishell.getDimension());
	dim::LogDimension dim_logger(screen);
	dim::setSize(&logger_, dim_logger.getDimension());

	/* create basic GUI items */
	createMenu();
	createToolbar();

	/* create the dynamic GUI area */
	control_panel_ = new MainPanel(this);
	this->setCentralWidget(control_panel_);
	setStatus("Ready");

	const QString xml_file("doc/examples/meteoio_config.xml");
	XMLReader xml;
	QString xml_error;
	const bool success = xml.read(xml_file, xml_error);
	if (!success)
		Error(tr("Errors occured when parsing the XML configuration file"),
		    tr("File: \"") + xml_file + "\"", xml_error);
	buildGui(xml.getXml());

	INIParser ini(logger_);
	ini.setFile("io.ini");
	QTextStream ss(stdout);
	ini.outputIni(ss);
	std::cout << "-----" << std::endl;
	ini.outputIni(ss, true);
}

/**
 * @brief Build the dynamic GUI.
 * This function initiates the recursive GUI building with an XML document that was
 * parsed beforehand.
 * @param[in] xml XML to build the gui from.
 */
void MainWindow::buildGui(const QDomDocument &xml)
{
	QDomNode root = xml.firstChild();
	 //no parent group - tabs will be created for top level:
	recursiveBuild(root, nullptr, QString());
}

/**
 * @brief Build the main window's menu items.
 */
void MainWindow::createMenu()
{
	/* File menu */
	QMenu *menu_file = this->menuBar()->addMenu(tr("&File"));
	auto *file_quit = new QAction(tr("&Exit"), this);
	menu_file->addAction(file_quit);
	connect(file_quit, &QAction::triggered, this, &MainWindow::quitProgram);

	/* View menu */
	QMenu *menu_view = this->menuBar()->addMenu(tr("&View"));
	auto *view_log = new QAction(tr("&Log"), this);
	menu_view->addAction(view_log);
	connect(view_log, &QAction::triggered, this, &MainWindow::viewLogger);
	auto *view_settings = new QAction(tr("&Settings"), this);
	menu_view->addAction(view_settings);
	connect(view_settings, &QAction::triggered, this, &MainWindow::viewSettings);
}

/**
 * @brief Create the main window's toolbar.
 */
void MainWindow::createToolbar()
{
	//TODO: store position
	toolbar_ = this->addToolBar("main");
	QPixmap file_open(":/icons/file_open.png");
	toolbar_->addAction(QIcon(file_open), tr("Open File"));
}

/**
 * @brief Display a text in the main window's status bar.
 * @details Events such as hovering over a menu may clear the status. It is meant
 * for temporary messages.
 * @param[in] message The text to display.
 * @param[in] time Time to display the text for.
 */
void MainWindow::setStatus(const QString &message, const int &time)
{ //only temporary messages are possible this way
	statusBar()->showMessage(message, time);
	//TODO: use a label that can stay visible and be styled
}

/**
 * @brief Event handler for the "File::Quit" menu: quit the main program.
 */
void MainWindow::quitProgram()
{
	QApplication::quit();
}

/**
 * @brief Event handler for the "View::Log" menu: open the Logger window.
 */
void MainWindow::viewLogger()
{
	logger_.show(); //the logger is always kept in scope
	logger_.raise(); //bring to front
}

/**
 * @brief Event handler for the "View::Settings" menu: open the settings dialog.
 */
void MainWindow::viewSettings()
{
	static auto *settings_dialog = new Settings(this); //allow only once, load when asked
	settings_dialog->show();
	settings_dialog->raise();
}
