#include "HelpWindow.h"
#include "AboutWindow.h"
#include "src/main/dimensions.h"
#include "src/main/inishell.h"
#include "src/main/XMLReader.h"

#include <QCoreApplication>
#include <QDesktopServices>
#include <QMenuBar>

#ifdef DEBUG
	#include <iostream>
#endif

HelpWindow::HelpWindow(QMainWindow *parent) : QMainWindow(parent)
{

	/* create a scroll area and put a Group in it */
	help_tab_ = new SectionTab();
	this->setCentralWidget(help_tab_);

	dim::setDimensions(this, dim::HELP);
	this->setWindowTitle(tr("Help") + " ~ " + QCoreApplication::applicationName());

	createMenu();
}

void HelpWindow::loadHelp(const bool &dev)
{
	static bool is_loaded = false;
	if (!is_loaded | dev) {
		help_tab_->clear();
		XMLReader xmlr;
		QString err;
		QString help_file( "help.xml" );
		if (dev)
			help_file = "help_dev.xml";
		xmlr.read(":doc/" + help_file, err);
		QDomDocument xml(xmlr.getXml());
		buildGui(xml, help_tab_);
		is_loaded = true && !dev; //reload after opening dev
	}

	view_dev_help_->setChecked(dev);
	view_user_guide_->setChecked(!dev);
	this->show();
	this->raise();
}

/**
 * @brief Event listener for the ESC key.
 * @details Close the help on pressing ESC.
 * @param event The key press event that is received.
 */
void HelpWindow::keyPressEvent(QKeyEvent *event)
{
	if (event->key() == Qt::Key_Escape || keyToSequence(event) == QKeySequence::Close)
		this->close();
}

void HelpWindow::createMenu()
{
	/* FILE menu */
	QMenu *menu_file = this->menuBar()->addMenu(tr("&File"));
	auto *file_bugreport_ = new QAction(tr("File &bug report..."), menu_file);
	menu_file->addAction(file_bugreport_);
	connect(file_bugreport_, &QAction::triggered,
		[]{ QDesktopServices::openUrl(QUrl("https://models.slf.ch/p/inishell-ng/issues/")); });
	menu_file->addSeparator();
	auto *file_close = new QAction(getIcon("application-exit"), tr("&Close"), menu_file);
	file_close->setShortcut(QKeySequence::Quit);
	file_close->setMenuRole(QAction::QuitRole);
	menu_file->addAction(file_close);
	connect(file_close, &QAction::triggered, this, [=]{ this->close(); });

	/* VIEW menu */
	QMenu *menu_view = this->menuBar()->addMenu(tr("&View"));
	view_user_guide_ = new QAction(tr("&User guide"), menu_view); //no icon to get checkboxes
	view_user_guide_->setCheckable(true);
	view_user_guide_->setChecked(true);
	menu_view->addAction(view_user_guide_);
	view_user_guide_->setShortcut(QKeySequence::HelpContents);
	connect(view_user_guide_, &QAction::triggered, this, [=]{ loadHelp(); });
	view_dev_help_ = new QAction(tr("&Developer's guide"), menu_view);
	view_dev_help_->setCheckable(true);
	menu_view->addAction(view_dev_help_);
	connect(view_dev_help_, &QAction::triggered, this, [=]{ loadHelp(true); });
	menu_view->addSeparator();
	auto *help_about = new QAction(getIcon("help-about"), tr("&About"), menu_view);
	help_about->setMenuRole(QAction::AboutRole);
	menu_view->addAction(help_about);
	connect(help_about, &QAction::triggered, this, &HelpWindow::helpAbout);
	help_about->setShortcut(QKeySequence::WhatsThis);
	auto *show_paper = new QAction(tr("Show &paper..."), menu_view);
	menu_view->addAction(show_paper);
	connect(show_paper, &QAction::triggered, this, [=]{ QDesktopServices::openUrl(QUrl(":doc/paper.pdf")); });
	show_paper->setEnabled(false);
}

void HelpWindow::helpAbout()
{
	static auto *about = new AboutWindow(this);
	about->show();
	about->raise();
}
